<?php
/**
 * Class Seo\BrowseContent
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Core\Models\Seo;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Mtc\Core\Taxonomy;

/**
 * Class Seo\BrowseContent
 *
 * Seo Browse Content model
 * Allows managing content displayed on browse pages for specific sections
 *
 * @package Mtc\Core
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class BrowseContent extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'seo_browse_content';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'description'
    ];

    /**
     * Relationship with Taxonomy bindings for this content page
     *
     * @return \Illuminate\Database\Eloquent\Relations\belongsToMany
     */
    public function taxonomies()
    {
        return $this->belongsToMany(
            Taxonomy::class,
            'seo_browse_content_taxonomies',
            'browse_content_page_id',
            'node_id'
        );
    }

    /**
     * Find if this browse page has content defined
     *
     * This finds a best match for the taxonomy selections against what is set in database
     *
     * @param Request $request incoming request
     * @return BrowseContent Content - filled or empty record if no content found
     */
    public static function getContentForSelection(Request $request)
    {
        if (count($request->input('selected'))) {
            // make sure we are working with collection (ajax vs page load)
            $selected = $request->get('selected');
            if (is_array($selected)) {
                $selected = collect($selected);
            }

            // Get all selected taxonomy ids
            $selected_taxonomy_ids = $selected->pluck('id');


            // Find all matching content entries
            $matching_browse_content = self::with('taxonomies')
                ->whereHas('taxonomies', function ($query) use ($selected_taxonomy_ids) {
                    return $query->whereIn('taxonomies.id', $selected_taxonomy_ids);
                })
                ->get()
                ->reject(function (self $browse_content) use ($selected_taxonomy_ids) {
                    // We need to discard those entries
                    // that have additional entries other than ones we have in selection
                    return $browse_content->taxonomies()
                        ->whereNotIn('taxonomies.id', $selected_taxonomy_ids)
                        ->count() > 0;
                })
                ->sortBy(function(self $browse_content) {
                    // We need to order them by most matching taxonomies
                    return $browse_content->taxonomies->count();
                }, SORT_REGULAR, true);

            // if we have successful matches return the best option
            if (count($matching_browse_content) > 0) {
                return $matching_browse_content->first();
            }
        }

        return new self;
    }
}
