<?php
/**
 * Class Coupon
 *
 * @package Mtc\Shop
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-04-14
 */

namespace Mtc\Coupons\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use \Closure;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Event;
use Mtc\Core\Taxonomy;
use Mtc\Coupons\Http\Controllers\CouponController;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\DiscountContract;
use Mtc\Coupons\Models\Coupon\Restriction;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Http\Controllers\CheckoutController;

/**
 * Class Coupon
 *
 * Coupon eloquent class.
 * Manages coupon functionality.
 *
 * @package Mtc\Shop
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-04-14
 */
class Coupon extends Model implements DiscountContract
{
    use SoftDeletes;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'coupons';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'code',
        'redemptions',
        'type',
        'value',
        'exclude_sale',
        'allow_multiple_discounts',
        'grant_free_delivery',
        'min_basket_amount',
        'only_first_order'
    ];

    /**
     * @var string[] $coupon_types Define existing coupon types
     */
    public static $coupon_types = [
        'amount_off' => 'Set Amount Discount',
        'percent_off' => 'Percentage Discount'
    ];

    /**
     * Relationship with restrictions to specific products / taxonomies
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function restrictions()
    {
        return $this->hasMany(Restriction::class, 'coupon_id');
    }

    /**
     * Relationship with restrictions that are allowed
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function allowedRestrictions()
    {
        return $this->restrictions()->where('is_exclude', 0);
    }

    /**
     * Relationship with restrictions to specific products / taxonomies
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function inclusiveRestrictions()
    {
        return $this->restrictions()
            ->where('is_exclude', 0);
    }

    /**
     * Relationship with restrictions to specific products / taxonomies
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function exclusiveRestrictions()
    {
        return $this->restrictions()
            ->where('is_exclude', 1);
    }

    /**
     * Register this class as a valid discount provider if config is enabled
     */
    public static function register()
    {
        if (config('shop.coupons_enabled')) {
            Event::listen('Discount::getList', function () {
                return self::class;
            });
        }
    }

    /**
     * @param Request $request incoming request
     * @param BasketContract $basket Basket instance
     */
    public function apply(Request $request, BasketContract $basket)
    {
        $discount = $basket->discounts()->where('discount_type', self::class)
            ->where('discount_id', $this->id)
            ->first();


        if (!$discount) {
            $this->increment('used_in_basket_count');
            $basket->discounts()->create([
                'discount_type' => self::class,
                'discount_id' => $this->id
            ]);

            $basket->load('discounts');
        }

        return $basket;
    }

    /**
     * Load all the functionality required for discount to function
     *
     * @param BasketContract $basket current basket
     * @param Closure $next closure for next discount in pipeline
     * @return mixed call the next pipeline discount and pass the basket to it
     */
    public static function loadDiscount(BasketContract $basket, Closure $next)
    {
        $visible = true;
        foreach ($basket->discounts as $basket_discount) {
            if ($basket_discount->discount_type === self::class) {
                // Since we need to validate the coupon code we need to add it to the request
                request()->request->add(['code' => $basket_discount->discount->code]);

                $validation_result = self::validate(request(), $basket);

                // If this is a valid coupon we attach the discount value and reference
                if ($validation_result instanceof self) {
                    $visible = false;
                    // Get the discount amounts
                    $amounts = $basket_discount->discount->getApplicableDiscountAmount($basket);
                    $basket_discount->amount_including_tax = (int)$amounts['amount'];
                    $basket_discount->amount_excluding_tax = (int)$amounts['amount_ex_vat'];
                    $basket_discount->tax_on_discount_amount = (int)abs($amounts['amount'] - $amounts['amount_ex_vat']);
                    $basket_discount->reference = $basket_discount->discount->code;
                } else {
                    // this discount is not valid so we need to remove it
                    $basket->discounts()
                        ->where('id', $basket_discount->id)
                        ->delete();

                    // Reload discounts
                    $basket->load('discounts');
                }

            }
        }

        $basket->discount_forms->push([
            'template' => 'coupons::public.checkout.voucher_form',
            'visible' => $visible,
            'data' => [
                'voucher_action' => action('\\' . CouponController::class . '@store'),
                'error' => !empty($validation_result) && is_string($validation_result) ? $validation_result : ''
            ]
        ]);

        return $next($basket);
    }

    /**
     * Validate if the submitted discount can be applied to current basket
     *
     * @param Request $request incoming request
     * @param BasketContract $basket current basket instance
     * @return Coupon|string coupon or error message
     */
    public static function validate(Request $request, BasketContract $basket)
    {
        // if we did not find a coupon code, return an error message
        if (empty($request->input('code'))) {
            return trans('coupons::errors.code_required');
        }

        /**
         * Lets find a valid coupon based on the given code and basket
         * @var Coupon $coupon
         */
        $coupon = self::where('code', $request->input('code'))
            ->where('date_from', '<=', Carbon::today())
            ->where('date_to', '>=', Carbon::today())
            ->where('redemptions', '>', 0)
            ->where('min_basket_amount', '<=', $basket->getCostSubtotalAttribute())
            ->first();

        // if we did not find a coupon code, return an error message
        if (!$coupon) {
            return trans('coupons::errors.invalid_code');
        }

        // If we have advanced coupons enabled we need to do additional checks
        if (config('coupons.coupons_advanced')) {
            // Check if discount stacking is not allowed and basket already has other discounts
            if (!$coupon->allow_multiple_discounts && count($basket->discounts) > 0) {
                // We still need to allow this discount in the basket
                if ($basket->discounts->where('discount_type', '!=', self::class)->count() == 1) {
                    return trans('coupons::errors.discount_stacking');
                }
            }

            // Check if user has already placed an order based on known info
            if ($coupon->only_first_order) {
                $query = resolve(OrderContract::class)::query();

                if (!empty($basket->email)) {
                    $query->where('email', $basket->email);
                }

                if ($basket->user_id) {
                    $query->where('email', $basket->user->email);
                }

                if ($query->count() > 0) {
                    return trans('coupons::errors.only_first_order');

                }
            }

            // Lastly - lets check if discounts apply to basket contents
            if (count($coupon->restrictions) > 0) {
                if (count($coupon->basketItemsMatchingRestrictions($basket)) == 0) {
                    return trans('coupons::errors.restricted_discount');
                }
            }
        }

        return $coupon;

    }

    /**
     * Calculate the applicable discount amount for the basket based on given discount
     * This takes into account the two supported discount types - set amount and percentage discounts
     *
     * @param BasketContract $basket current basket
     * @return float[] discount amount
     */
    public function getApplicableDiscountAmount(BasketContract $basket)
    {
        if ($this->type === 'amount_off') {
            // If coupon has restricted items / taxonomies we need to filter those out from item cost
            if (config('coupons.coupons_advanced') && count($this->restrictions)) {
                // Get all items that match restrictions
                $items = $this->basketItemsMatchingRestrictions($basket);

                // We need to set the prices in basket for all items
                $items->map(function ($item) {
                    // Loop through both including and excluding tax
                    foreach (['price_excluding_tax', 'price_including_tax'] as $key) {
                        $item->{$key} = $item->variant
                            ->prices
                            ->multiple($item->quantity)
                            ->{$key};
                    }
                    return $item;
                });

                // Return the discount amount applicable
                return [
                    'amount' => $items->sum('price_including_tax') < $this->value ? $items->sum('price_including_tax') : $this->value,
                    'amount_ex_vat' => $items->sum('price_excluding_tax') < $this->value ? $items->sum('price_excluding_tax') : $this->value,
                ];
            }

            /*
             * $discount_without_tax covers the situation where discount is smaller than the subtotal
             * Since products can have varying tax rates and discount must be spread evenly across all products
             * we find the ratio between sub-total and sub-total without tax
             * and get the discount amount without tax based on this ratio
             */
            $discount_without_tax = $this->value
                * $basket->getCostSubtotalWithoutTaxAttribute() / $basket->getCostSubtotalAttribute();

            // No additional restrictions, all we need to do is
            // ensure discount amount is not higher than basket amount
            return [
                'amount' => $basket->getCostSubtotalAttribute() < $this->value ?
                    $basket->getCostSubtotalAttribute() : $this->value,
                'amount_ex_vat' => $basket->getCostSubtotalAttribute() < $this->value ?
                    $basket->getCostSubtotalWithoutTaxAttribute() : $discount_without_tax,
            ];

        }

        if ($this->type === 'percent_off')  {
            // If coupon has restricted items / taxonomies we need to filter those out from item cost
            if (config('coupons.coupons_advanced') && count($this->restrictions)) {
                // Get all items that match restrictions
                $items = $this->basketItemsMatchingRestrictions($basket);


                // We need to set the prices in basket for all items
                $items->map(function ($item) {
                    // Loop through both including and excluding tax
                    foreach (['price_excluding_tax', 'price_including_tax'] as $key) {
                        $item->{$key} = $item->variant
                            ->prices
                            ->multiple($item->quantity)
                            ->{$key};
                    }
                    return $item;
                });

                // Return the discount amount applicable
                // 0.01 * 0.01
                // Once for percentage, once for the fact that upon saving we are converting to pennies
                return [
                    'amount' => $items->sum('price_including_tax') * 0.01 * 0.01 * $this->value,
                    'amount_ex_vat' => $items->sum('price_excluding_tax') * 0.01 * 0.01 * $this->value,
                ];
            }

            // No additional restrictions, all we need to do is
            // ensure discount amount is not higher than basket amount
            // 0.01 * 0.01
            // Once for percentage, once for the fact that upon saving we are converting to pennies
            return [
                'amount' => $basket->getCostSubtotalAttribute() * 0.01 * 0.01 * $this->value,
                'amount_ex_vat' => $basket->getCostSubtotalWithoutTaxAttribute() * 0.01 * 0.01 * $this->value
            ];
        }

        // Fallback to 0 values
        return [
            'amount' => 0,
            'amount_ex_vat' => 0

        ];
    }

    /**
     * Find all basket items that match the restrictions set to current coupon instance
     *
     * This script checks if the basket contents match the set restrictions on the basket.
     * If no items match the restrictions set the discount is not valid
     *
     * @param BasketContract $basket current basket
     * @return Collection list of applicable items
     */
    public function basketItemsMatchingRestrictions(BasketContract $basket): Collection
    {
        $restrictions = $this->restrictions;
        return $basket->items
            ->filter(function ($item) use ($restrictions) {
                // If there are restrictions that are marked as inclusive we need to make sure
                // that all of these match instead. If something doesn't match we need to discard item
                if ($restrictions->where('is_exclude', 0)->count() > 0) {
                    // Check if item is in allowed list
                    $allowed_item = $restrictions->where('restriction_type', '\\' . get_class($item->variant->product))
                        ->where('is_exclude', 0)
                        ->where('restriction_id', $item->variant->product_id);

                    // if item is allowed allow returning item
                    if (count($allowed_item)) {
                        return true;
                    }

                    // Check if item taxonomy is in allowed list
                    $allowed_taxonomies = $restrictions->where('is_exclude', 0)
                        ->where('restriction_type', '\\' . get_class($item->variant->product))
                        ->where('restriction_id', $item->variant->product->node->taxonomies->pluck('id'));

                    // if item is allowed allow returning item
                    if (count($allowed_taxonomies)) {
                        return true;
                    }

                    return false;
                }

                $restricted_item_count = $restrictions->where('is_exclude', 1)
                    ->where('restriction_type', '\\' . get_class($item->variant->product))
                    ->where('restriction_id', $item->variant->product_id)
                    ->count();

                // If a restricted item was found we discard this item
                if ($restricted_item_count) {
                    return false;
                }

                $restricted_taxonomy_count = $restrictions->where('restriction_type', '\\' . Taxonomy::class)
                    ->where('is_exclude', 1)
                    ->whereIn('restriction_id', $item->variant->product->node->taxonomies->pluck('id'))
                    ->count();

                // If a restricted taxonomies were found we discard this item
                if ($restricted_taxonomy_count) {
                    return false;
                }

                // No restrictions hit, allow to return this item
                return true;

            });
    }

    /**
     * When order is processed we need to update coupon.
     * Redemption count needs to be reduced and purchase count needs to be increased if coupons was used
     * @param OrderContract $order
     */
    public static function processOrder(OrderContract $order)
    {
        $coupon_discounts = $order->discounts->where('discount_type', self::class);
        if (count($coupon_discounts)) {
            foreach ($coupon_discounts as $discount) {
                /**
                 * @var self $coupon discount is morphed to this class since we filtered them out earlier
                 */
                $coupon = $discount->discount;
                $coupon->increment('purchase_count');
                $coupon->decrement('redemptions');
            }
        }
    }

}