<?php
/**
 * Coupon Service Provider
 *
 * Register views and translations for the current component.
 *
 * @package Mtc\Coupons
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Coupons\Providers;

use Illuminate\Support\Facades\Event;
use Illuminate\Routing\Router;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Mtc\Core\Admin\Menu;
use Mtc\Core\Events\Admin\MenuLoading;
use Mtc\Coupons\Http\Controllers\Admin\CouponController;
use Mtc\Coupons\Http\Controllers\CouponController as PublicCouponController;
use Mtc\Coupons\Models\Coupon;
use Mtc\Shop\Events\OrderPaid;

/**
 * Connect the component to Laravel
 *
 * @package Mtc\Coupons
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class CouponServiceProvider extends ServiceProvider
{
    /**
     * Boot the service provider.
     */
    public function boot()
    {
        $this->defineResources();
        $this->defineRoutes();
        $this->definePublished();
    }

    /**
     * Register the service provider
     * This method is run before boot so events should be dropped here
     */
    public function register()
    {
        $this->defineEvents();
    }

    /**
     * Define the events for this component.
     */
    protected function defineEvents()
    {
        if (config('shop.coupons_enabled')) {
            Event::listen(MenuLoading::class, function () {
                Menu::addSubMenu(
                    'discounts',
                    trans('coupons::titles.manage_coupons'),
                    '',
                    action('\\' . CouponController::class . '@index'),
                    'manage_coupons'
                );
            });

            // Add Coupons to discount provider list
            Event::listen('shop.discount_providers', function () {
                return Coupon::class;
            });

            // Make sure we update coupon on completed order
            Event::listen(OrderPaid::class, function (OrderPaid $event) {
                Coupon::processOrder($event->order);
            });

            Event::listen('shop.shipping_methods', function () {
                return new \Mtc\Coupons\Models\VoucherDelivery;
            });

        }

    }

    /**
     * Define routes and wildcards used by this component
     */
    protected function defineRoutes()
    {
        // define mapping for {coupon} to auto-convert to a Coupon instance
        Route::model('coupon', Coupon::class);

        // Set up routes used by the component
        if (!$this->app->routesAreCached()) {
            Route ::group([
                'namespace' => '\Mtc\Coupons\Http\Controllers',
                'middleware' => ['web'],
                'as' => 'coupons.'
            ], function (Router $router) {

                $router->post('checkout/discounts/coupons', 'CouponController@store');

                // Admin routes
                $router->group([
                    'prefix' => 'admin/discounts',
                    'middleware' => [ 'auth', 'permission:admin'],
                    'as' => 'admin.'
                ], function (Router $router) {
                    // Coupon routes
                    $router->post('coupons/restrictions', 'Admin\CouponController@getRestrictionOptions');
                    $router->post('coupons/{coupon}/remove_restriction', 'Admin\CouponController@removeRestriction');
                    $router->post('coupons/{coupon}/restrictions', 'Admin\CouponController@setRestrictions');
                    $router->resource('coupons', 'Admin\CouponController');
                }
                );
            });

        }
    }

    /**
     * Define the resources for this package.
     */
    protected function defineResources()
    {
        $component_path = dirname(dirname(__DIR__));
        $this->loadViewsFrom($component_path . '/resources/views', 'coupons');
        $this->loadTranslationsFrom($component_path . '/resources/lang', 'coupons');
    }

    /**
     * Register the publishable resources for this component
     * This registers that the component has migrations and configs that should be copied over
     * via php artisan vendor:publish
     */
    protected function definePublished()
    {
        $this->publishes([
            __DIR__ . '/../../config/' => config_path()
        ], 'config');

        $this->publishes([
            __DIR__ . '/../../database/migrations/' => database_path('migrations')
        ], 'migrations');
    }
}
