<?php
/**
 * Class VoucherDelivery
 *
 * @package Mtc\Coupons
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-17
 */

namespace Mtc\Coupons\Models;

use Illuminate\Support\Collection;
use Mtc\Shop\Abstracts\ShippingMethod;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Http\Controllers\Admin\ShippingFlatRateController;
use Mtc\Shop\ShippingFlatRate;

/**
 * Class VoucherDelivery
 *
 * Free delivery method provided by voucher
 *
 *
 * @package Mtc\Coupons
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017-05-17
 */
class VoucherDelivery extends ShippingMethod
{
    /**
     * @var string $id Shipping Method identificator
     */
    public $id = 'voucher-delivery';

    /**
     * @var string $title Shipping Method title
     */
    public $title = 'Free Voucher Delivery';

    /**
     * Calculate shipping costs
     *
     * @param BasketContract|null $basket
     * @return array
     */
    public function calculateShipping(BasketContract $basket = null): array
    {
        // No basket given means we can't give any rates
        // Also not applicable to baskets without discounts
        if (empty($basket) || count($basket->discounts) == 0) {
            return [];
        }

        $coupon_discounts = $basket->discounts
            ->where('discount_type', Coupon::class);

        if ($coupon_discounts->count() == 0) {
            return [];
        }

        $has_free_delivery = $coupon_discounts->map(function ($discount) use ($basket)  {
            // Free delivery everywhere
            if ($discount->discount->grant_free_delivery == 1) {
                return [
                    'value' => 0,
                    'id' => 'free',
                    'title' => 'Free Delivery'
                ];
            }

            // Local free delivery & delivery to current address
            if ($discount->discount->grant_free_delivery == 2
                && $basket->shippingAddress->country == config('core.default_country', 'GB')
            ) {
                return [
                    'value' => 0,
                    'id' => 'free',
                    'title' => 'Free Delivery'
                ];
            }
        })->toArray();

        return $has_free_delivery ?: [];
    }

    /**
     * Flat rates always can be edited/added
     *
     * @return string
     */
    public function isManageable()
    {
        return false;
    }

}
