<?php
/**
 * Coupon Controller
 *
 * @package mtc/coupons
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017.04.08
 */

namespace Mtc\Coupons\Http\Controllers\Admin;

use Carbon\Carbon;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Coupons\Models\Coupon;
use Mtc\Coupons\Http\Requests\Admin\StoreCoupon;
use Mtc\Shop\Order\Order;

/**
 * Coupon Controller
 *
 * This controller is used for managing coupons on site
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version 2017.04.08
 */
class CouponController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('coupons.admin.coupons', Coupon::query(), 'coupons::titles.coupons'))
            ->columns(
                [
                    'name' => trans('fields.name'),
                    'code' => trans('coupons::fields.code'),
                    'redemptions' => trans('coupons::fields.redemptions'),
                    'type' => trans('coupons::fields.type'),
                    'value' => trans('coupons::fields.value'),
                    'date_from' => trans('coupons::fields.date_from'),
                    'date_to' => trans('coupons::fields.date_to'),
                    'used_in_basket_count' => trans('coupons::fields.basket_count'),
                    'purchase_count' => trans('coupons::fields.order_count'),
                ]
            )
            ->data([
                'value' => function ($coupon) {
                    return $coupon->value / 100;
                }
            ])
            ->view();
    }
    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $coupon = new Coupon();
        $resource_name = 'coupons.admin.coupons';
        $action = route(...[
            "{$resource_name}.store",
            [$coupon->id]
        ]);
        return view('core::admin.builder.item')
            ->with([
                'title' => 'coupons::titles.coupon',
                'view' => 'coupons::admin.coupons.form',
                'name' => $resource_name,
                'item' => $coupon,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Create a new Coupon
     *
     * @param StoreCoupon $request Request with validation
     * @param Coupon $coupon Empty coupon model
     *
     * @return \Illuminate\Http\Response Redirect to seo default listing
     */
    public function store(StoreCoupon $request, Coupon $coupon)
    {
        $add = empty($coupon->id);
        $coupon->fill($request->all());
        $coupon->value = $request->input('value') * 100;
        $coupon->min_basket_amount = $request->input('min_basket_amount') * 100;
        $coupon->date_from = strtotime($request->input('date_from')) > 0 ? $request->input('date_from') : Carbon::now()->toDateString();
        $coupon->date_to = strtotime($request->input('date_to')) > 0 ? $request->input('date_to') : Carbon::now()->addYear()->toDateString();
        $coupon->save();

        if ($add) {
            $request->session()->flash('success', trans('coupons::titles.coupon_added') . $coupon->name);
        } else {
            $request->session()->flash('success', trans('coupons::titles.coupon_updated') . $coupon->name);
        }
        return redirect(route('coupons.admin.coupons.index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Coupon $coupon Empty coupon model
     * @return \Illuminate\View\View
     */
    public function edit(Coupon $coupon)
    {
        // Load restriction relationships
        $coupon->load(['inclusiveRestrictions.restriction', 'exclusiveRestrictions.restriction']);
        $resource_name = 'coupons.admin.coupons';

        // Action for updating coupon
        $action = route(...[
            "{$resource_name}.update",
            [$coupon->id]
        ]);

        // Action for restriction management
        $restriction_action = route(...[
            "{$resource_name}.index"
        ]);
        return view('core::admin.builder.item')
            ->with([
                'title' => 'coupons::titles.coupon',
                'view' => 'coupons::admin.coupons.form',
                'name' => $resource_name,
                'item' => $coupon,
                'node' => null,
                'form_action' => $action,
                'restriction_action' => $restriction_action,
                'form_method' => 'PUT',
                'route_params' => null,
                'restriction_types' => Coupon\Restriction::$supported_restrictions
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreCoupon $request Request with validation
     * @param Coupon $coupon Empty coupon model
     * @return \Illuminate\Http\Response
     */
    public function update(StoreCoupon $request, Coupon $coupon)
    {
        return $this->store($request, $coupon);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param Coupon $coupon Coupon to delete
     */
    public function destroy(Request $request, Coupon $coupon)
    {
        $coupon->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Coupon '{$coupon->name}' has been deleted.");
        return redirect()->back();
    }

    /**
     * Get all restriction options for object type
     *
     * @param Request $request
     * @return \Illuminate\Database\Eloquent\Collection|array
     */
    public function getRestrictionOptions(Request $request)
    {
        $instance_type = $request->input('type');

        // Make sure we can fetch all entries
        if (class_exists($instance_type) && array_key_exists($instance_type, Coupon\Restriction::$supported_restrictions)) {
            return $instance_type::all();
        }
        return [];

    }

    /**
     * Set restrictions for a coupon
     *
     * @param Request $request
     * @param Coupon $coupon
     * @return Coupon updated coupon
     */
    public function setRestrictions(Request $request, Coupon $coupon)
    {
        // Make sure we have we have a list of items
        if (!is_array($request->input('items'))) {
            return $coupon;
        }

        foreach ($request->input('items') as $item) {
            $coupon->restrictions()
                ->firstOrCreate([
                    'restriction_type' => $request->input('type'),
                    'is_exclude' => (int)$request->input('isExclusive'),
                    'restriction_id' => $item['id']
                ]);
        }

        // Reload relationship and return result
        $coupon->load(['exclusiveRestrictions.restriction', 'inclusiveRestrictions.restriction']);
        return $coupon;
    }

    /**
     * Delete restriction for coupon
     *
     * @param Request $request incoming request
     * @param Coupon $coupon current coupon
     * @return Coupon updated coupon
     */
    public function removeRestriction(Request $request, Coupon $coupon)
    {
        // Remove the restriction if the id is found in current coupons restriction list
        Coupon\Restriction::where('coupon_id', $coupon->id)
            ->where('id', $request->input('remove_id'))
            ->delete();

        // Reload relationship and return result
        $coupon->load(['exclusiveRestrictions.restriction', 'inclusiveRestrictions.restriction']);
        return $coupon;
    }
}
