<?php

namespace Mtc\Components\Languages\Http\Controllers\Admin;

use Event;
use Illuminate\Support\Facades\File;
use Mtc\Components\Languages\Language;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;

/**
 * Displays the language routes as a resource
 *
 * @category Mtc\Components\Languages\Http\Controllers\Admin
 * @package  Mtc\Components\Languages
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class LanguageController extends Controller
{
    /**
     * Ensure this is only accessible to those with the permission to do so.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:manage-languages');
        $this->middleware('permission:add-delete-languages', [
            'only' => [
                'create',
                'store',
                'delete'
            ]
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View
     */
    public function index()
    {
        return (new Builder('languages.admin.languages', Language::query()))
            ->setAddPermission('add-delete-languages')
            ->setDeletePermission('add-delete-languages')
            ->columns([
                'name' => trans('fields.name'),
                'code' => trans('core::fields.code'),
                'enabled' => trans('core::fields.is_active')
            ])
            ->data([
                'name' => function ($item) {
                    return $item->name;
                },
                'code' => function ($item) {
                    return $item->locale_code;
                },
                'enabled' => function ($item) {
                    return $item->enabled ? trans('Yes') : trans('No');
                }
            ])
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {

        $language = new Language();
        // Set the default action to create an item
        $action = route("languages.admin.languages.store", [
            $language->id
        ]);

        $method = 'POST';
        return view('languages::admin.languages.edit')
            ->with([
                'title' => 'Language',
                'item' => $language,
                'form_action' => $action,
                'form_method' => $method,
                'route_params' => ['language' => $language->id],
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @param Language $language
     *
     * @return \Illuminate\Http\Response Redirect to edit page
     */
    public function store(Request $request, Language $language)
    {
        $this->validate($request, [
            'name' => 'required',
            'locale_code' => 'required|unique:languages',
        ]);
        try {
            $language->fill($request->input());
            $language->save();
        } catch (\Exception $exception) {
            session()->flash('warning', $exception->getMessage());
        }

        session()->flash('success', 'Language created');
        return redirect(route('languages.admin.languages.edit', [
            'language' => $language->id
        ]));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Language $language
     * @return \Illuminate\View\View
     */
    public function edit(Language $language)
    {
        // Set the default action to create an item
        $action = route("languages.admin.languages.update", [
            'language' => $language->id
        ]);

        $method = 'PUT';
        return view('languages::admin.languages.edit')
            ->with([
                'title' => 'Language',
                'item' => $language,
                'form_action' => $action,
                'form_method' => $method,
                'route_params' => ['language' => $language->id],
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }

    public function update(Request $request, Language $language)
    {
        $this->validate($request, [
            'name' => 'required',
        ]);

        $language_file = $request->file('language_file');
        $language->fill($request->input());
        $language->save();

        if (!empty($language_file)) {
            try {
                $file_content = File::get($language_file->getRealPath());
                $language->setTranslationFileData($file_content);
                session()->flash('success', 'Language File Uploaded!');
            } catch (\Exception $exception) {
                session()->flash('error', 'Failed to update uploaded language file. Make sure it is valid format');

            }
        }

        session()->flash('success', 'Language updated');
        return redirect(route('languages.admin.languages.edit', [
            'language' => $language->id
        ]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param Language $language
     * @return \Illuminate\Http\Response Either JSON success or redirect back
     */
    public function destroy(Request $request, Language $language)
    {
        $language->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "{$language->name} has been deleted.");
        return redirect()->back();
    }

    /**
     * Download current translation file for a specific language
     *
     * @param Request $request
     * @param Language $language
     * @param $code
     * @return \Illuminate\Http\Response
     */
    public function download(Language $language)
    {
        $data = json_encode($language->getFlatTranslations(), JSON_PRETTY_PRINT);

        return response()->make($data, 200, [
            'Content-type' => 'text/plain; charset=utf-8',
            'Content-Disposition' => sprintf('attachment; filename="%s"', $language->locale_code . '.json'),
            'Content-Length' => strlen($data)
        ]);
    }

}
