<?php
namespace Mtc\Components\Languages;

use Illuminate\Contracts\Translation\Loader;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

/**
 * Class Language
 * Manages available languages on site
 *
 * @package Mtc\Components\Languages
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Language extends Model
{
    /**
     * Set the name for the default translation namespace.
     * If the translation does not come from a vendor package
     * it will be prefixed with this namespace when exporting
     */
    const DEFAULT_NAMESPACE = 'app';

    /**
     * @var string[] variables that are mass assignable
     */
    protected $fillable = [
        'name',
        'locale_code',
        'enabled'
    ];

    /**
     * Filter only enabled languages
     *
     * @param $query
     * @return mixed
     */
    public function scopeEnabled($query)
    {
        return $query->where('enabled', 1);
    }

    public static function boot()
    {
        parent::boot();

        static::created(function (self $language) {
            $language->createJsonTranslationFile();
        });

        static::deleted(function (self $language) {
            $language->deleteJsonTranslationFile();
        });
    }


    /**
     * Generate the url of the same page in the given language
     *
     * @return string url with this page on different language
     */
    public function linkThisPage()
    {
        $locale_switch_type = config('locale.switch_type', 'url_param');
        switch ($locale_switch_type) {
            case 'subdomain':

                break;

            case 'url_prefix':

                break;

            case 'url_param':
            default:

                return request()->fullUrlWithQuery([
                    'lang' => $this->locale_code
                ]);
                break;
        }
        return request()->fullUrl();
    }

    /**
     * Process the request and set the expected language in session
     *
     * @param  \Illuminate\Http\Request $request
     */
    public static function processLanguageFromRequest($request)
    {
        // TODO: Add support for different types of language configuration - sub-domain & route defined options should be supported

        if ($request->input('lang') && Language::query()->enabled()->where('locale_code', $request->input('lang'))->exists()) {
            session()->put('language_selected', $request->input('lang'));
        }

    }

    /**
     * Load all translations for this locale
     * Translations returned in tree structure
     * Used mainly for translation route that serves VUe translations
     *
     * @return static
     */
    public function getTranslations()
    {
        $translations = $this->loadJsonTranslations();
        if (empty($translations)) {
            $translations = $this->getFileTreeTranslations();
        } else {
            $translations = self::convertFlatToTree($translations);
        }

        return $translations;
    }

    /**
     * Get the translations from all loaded files for this language
     * Used when JSON file is not present
     * Data is returned in tree structure
     *
     * @return static
     */
    private function getFileTreeTranslations()
    {

        /** @var Loader $translation_loader Translation loader (FileLoader by default) */
        $translation_loader = app('translator')->getLoader();

        /*
         * Load translations from vendor packages
         * This is done by taking the registered vendor translations and fetching their information
         * Lastly we strip out any empty sections that don't have any translation entries
         */
        $translations = collect($translation_loader->namespaces())
            ->map(function ($namespace_path, $namespace) use ($translation_loader) {
                /*
                 * We loop here through all language groups registered for this namespace
                 * Each group is keyed by its filename
                 * and mapped to return the entries within this file.
                 *
                 * The groups are found using global vendor path but since we are publishing these vendor files
                 * the loader will pick them up instead of the original vendor versions
                 */
                return collect(glob($namespace_path . "/{$this->locale_code}/*"))
                    ->keyBy(function ($path) {
                        return pathinfo($path, PATHINFO_FILENAME);
                    })
                    ->map(function ($path, $group) use ($translation_loader, $namespace) {
                        return $translation_loader->load($this->locale_code, $group, $namespace);
                    });
            })
            ->reject(function ($collection) {
                return $collection->isEmpty();
            });

        /*
         * We also need to add the root-level translations in.
         * These are stored in resource language section
         * We take all files in the locale and load their data
         * Loaded data is attached to the translation collection
         */
        collect(glob(resource_path('lang') . "/{$this->locale_code}/*"))
            ->each(function ($path) use ($translation_loader, $translations) {
                $group = pathinfo($path, PATHINFO_FILENAME);
                $values = $translation_loader->load($this->locale_code, $group);
                $translations->put($group, $values);
            });

        return $translations;
    }

    /**
     * Load all translations for this locale in flat list
     * Get all translations in a flat list that then can be downloaded to a json file
     * This primarily used in language download functionality
     *
     * @param boolean $override
     * @return static
     */
    public function getFlatTranslations($override = false)
    {
        /** @var Loader $translation_loader Translation loader (FileLoader by default) */
        $translation_loader = app('translator')->getLoader();

        $translations = collect($this->loadJsonTranslations());

        // Go through all registered translation namespaces
        collect($translation_loader->namespaces())
            ->each(function ($namespace_path, $namespace) use ($translation_loader, $translations, $override) {

                // Go through all translation groups (files) in this namespace
                // and key them by filename (group name)
                collect(glob($namespace_path . "/{$this->locale_code}/*"))
                    ->keyBy(function ($path) {
                        return pathinfo($path, PATHINFO_FILENAME);
                    })
                    ->each(function ($path, $group) use ($translation_loader, $namespace, $translations, $override) {

                        // Load translations
                        $data = $translation_loader->load($this->locale_code, $group, $namespace);

                        // make sure data is flat before adding entries to translation list
                        self::flattenFileData($data)
                            ->each(function ($entry, $key) use ($namespace, $group, $translations, $override) {
                                if (!$override && $translations->has("{$namespace}::{$group}.{$key}")) {
                                    return;
                                }
                                $translations->put("{$namespace}::{$group}.{$key}", $entry);
                            });
                    });
            });

        /*
         * We also need to add the root-level translations in.
         * These are stored in resource language section
         * We take all files in the locale and load their data
         * Loaded data is attached to the translation collection
         */
        collect(glob(resource_path("lang/{$this->locale_code}/*")))
            ->each(function ($path) use ($translation_loader, $translations, $override) {
                $group = pathinfo($path, PATHINFO_FILENAME);
                $data = $translation_loader->load($this->locale_code, $group);

                self::flattenFileData($data)
                    ->each(function ($entry, $key) use ($group, $translations, $override) {
                        if (!$override && $translations->has(self::DEFAULT_NAMESPACE . "::{$group}.{$key}")) {
                            return;
                        }
                        $translations->put(self::DEFAULT_NAMESPACE . "::{$group}.{$key}", $entry);
                    });

            });

        return $translations;
    }

    /**
     * Convert flat tree to tree structure
     * Takes flat tree, breaks down separators and creates a tree structure
     * Usage example - generate tree structure from JSON file for Vue translations
     *
     * @param array $data
     * @return string[]
     */
    private static function convertFlatToTree($data = [])
    {
        $tree = [];

        collect($data)->each(function ($entry, $key) use (&$tree) {
            // First level separation is namespace - it is separated by double comma ::
            // next levels are are separated by dots .
            $first_level_keys = explode('::', $key);
            $second_level_keys = explode('.', end($first_level_keys));

            // If there was a namespace defined we prepend it to the key list
            if (count($first_level_keys) == 2) {
                array_unshift($second_level_keys, $first_level_keys[0]);
            }

            /*
             * Here we loop through all key levels and break them down into tree
             * This tree contains one leaf for the translation tree
             * The leaf then is merged with the whole tree
             */
            $item = $entry;
            while (!empty($second_level_keys)) {
                $this_key_level = array_pop($second_level_keys);
                $item = [
                    $this_key_level => $item
                ];
            }

            $tree = array_merge_recursive($tree, $item);

        });

        return $tree;
    }

    /**
     * Flatten information listed in translation files
     * Takes tree structure and flattens it to single level to allow storing JSON file
     * Used in json file export
     *
     * @param array $data
     * @param string $prefix
     * @return \Illuminate\Support\Collection
     */
    private static function flattenFileData($data = [], $prefix = '')
    {
        $flat_map = collect([]);

        collect($data)->each(function ($entry, $key) use ($flat_map, $prefix) {
            if (is_array($entry)) {
                self::flattenFileData($entry, "{$prefix}{$key}.")
                    ->each(function ($flat_entry, $flat_key) use ($flat_map) {
                        $flat_map->put($flat_key, $flat_entry);
                    });
            } else {
                $flat_map->put($prefix . $key, $entry);
            }
        });

        return $flat_map;
    }

    /**
     * Save the JSON data to locale file
     * Updates the language translation file from admin
     *
     * @param string $json_data
     * @throws \Exception
     */
    public function setTranslationFileData($json_data)
    {
        if (json_decode($json_data) === false) {
            throw new \Exception('Uploaded File is not valid JSON format!');
        }

        $data_file = $this->getPathToJsonTranslations();
        File::put($data_file, $json_data);

        Cache::forget('translations_' . $this->locale_code);
    }

    /**
     * Get the list of existing JSON translations
     * Loads the JSON file and returns its content
     *
     * @param string $locale
     * @return mixed
     */
    private function loadJsonTranslations($locale = false)
    {
        if (!$locale) {
            $locale = $this->locale_code;
        }

        if (File::exists($this->getPathToJsonTranslations($locale))) {
            return json_decode(File::get($this->getPathToJsonTranslations($locale)), true) ?: [];
        }
        return [];
    }

    /**
     * Retrieve the path to JSON translation file for this locale
     *
     * @param string $locale
     * @return string
     */
    private function getPathToJsonTranslations($locale = false)
    {
        if (!$locale) {
            $locale = $this->locale_code;
        }
        return resource_path("lang/{$locale}.json");
    }

    /**
     * Create the translation file for language when creating a new language
     *
     * @throws \Exception
     */
    private function createJsonTranslationFile()
    {
        if (File::exists($this->getPathToJsonTranslations())) {
            return;
        }

        if ($this->locale_code !== config('app.fallback_locale') && !File::exists($this->getPathToJsonTranslations(config('app.fallback_locale')))) {
            throw new \Exception('Fallback locale is not set up');
        }

        $fallback_locale = $this->loadJsonTranslations(config('app.fallback_locale'));
        $this->setTranslationFileData(json_encode($fallback_locale));

        $overridden_translations = $this->getFlatTranslations(true);
        $this->setTranslationFileData(json_encode($overridden_translations));
    }

    /**
     * Delete the translation file for language when deleting a language
     *
     * @throws \Exception
     */
    private function deleteJsonTranslationFile()
    {
        if (File::exists($this->getPathToJsonTranslations())) {
            File::delete($this->getPathToJsonTranslations());
        }
    }
}
