<?php

namespace Mtc\Media\Http\Controllers\Admin;

use Illuminate\Support\Facades\Storage;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Media\Media;
use Mtc\Shop\Http\Requests\StoreImage;
use Mtc\Shop\Product;

/**
 * Actions concerning managing Media files
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class MediaController extends Controller
{

    /**
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public function show(Request $request)
    {
        if (is_int($request->input('model_id'))) {
            $image = Media::query()
                ->firstOrNew([
                    'parent_type' => $request->input('model'),
                    'parent_id' => $request->input('model_id'),
                ]);
            $image->setDefaultAttributes();
        }

        return [
            'media' => $image ?? null,
            'sizes' => Media::getSizesForModel($request->input('model'))
        ];

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id Media ID
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $media = Media::query()->find($request->input('media'));
        if ($media) {
            $media->delete();
        }

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Media file '{$media->title}' has been deleted.");
        return redirect()->back();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public function list(Request $request)
    {
        return [
            'media' => Media::query()
                ->where('parent_type', $request->input('model'))
                ->where('parent_id', $request->input('model_id'))
                ->get()
                ->map(function (Media $media) {
                    $media->setDefaultAttributes();
                    return $media;
                }),
            'sizes' => Media::getSizesForModel($request->input('model'))
        ];

    }

    /**
     * File upload processing
     *
     * @param Request $request
     * @return mixed
     */
    public function upload(Request $request)
    {
        // Make sure we let through requests with models passed
        $this->validate($request, [
            'model' => 'required',
            'file' => 'image'
        ]);

        $class_name = $request->input('model');

        // Find the upload path for the product image
        $upload_path = config('media.' . $class_name . '.upload_path', '');

        // Save the image in this location
        $path = Storage::disk('public')->put($upload_path, $request->file('file'));

        // Create media object
        $media = new Media([
            'src' => $path,
            'title' => $request->file('file')->getClientOriginalName(),
            'type' => $request->file('file')->getMimeType(),
            'parent_type' => $class_name
        ]);

        $class_name = $request->input('model');
        $object = (new $class_name)->find($request->input('model_id'));

        if ($object) {
            // Save this media file for the product
            $object->media()->save($media);

            if ($object->node) {
                $media->setMetaData($object->node, $request->file('file')->getClientOriginalExtension());
            }

            $media->save();
            $media->setDefaultAttributes();
            return $media;
        }

        // Add a unique seo friendly name for the image
        $media->url_name = time() . $request->file('file')->getClientOriginalName();
        $media->title = $request->file('file')->getClientOriginalName();
        $media->save();
        $media->setDefaultAttributes($request->input('thumb_size', 'thumb'));
        return $media;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreImage $request Validated Request
     * @param Product $product Product to attach media to
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        if ($request->input('action') === 'update_meta') {
            return $this->update($request);
        }

    }

    /**
     * Update information for a Media file
     *
     * @param Request $request incoming request
     * @param Product $product Product to whom media files are assigned to
     */
    public function updateMeta(Request $request, Product $product)
    {
        foreach ($request->input('media') as $key => $object) {
            $this->validate($request, [
                "media.{$key}.url_name" => 'required|alpha_dash_file|unique:media,url_name,' . $object['id']
            ], [], [
                "media.{$key}.url_name" => 'Image'
            ]);
        }

        // Update image url name and title
        foreach ($request->input('media', []) as $key => $media_request) {

            // Load the media file
            $media = Media::find($media_request['id']);

            if ($media) {
                // update values
                $media->url_name = $media_request['url_name'];
                $media->title = $media_request['title'];
                $media->save();
            }
        }
    }

    /**
     * Update order of images upon re-order
     *
     * @param Request $request incoming request
     * @param Product $product product whose images are in request
     */
    public function changeOrder(Request $request)
    {
        foreach ($request->input('media') as $order => $media) {
            Media::where('id', $media['id'])
                ->where('parent_type', $media['parent_type'])
                ->update([
                    'order' => $order
                ]);
        }
    }
}
