<?php
/**
 * Menu Controller (Admin)
 *
 * PHP Version 7
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus\Http\Controllers\Admin;

use Event;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Menus\Menu;
use Mtc\Menus\Http\Requests\Admin\StoreMenu;

/**
 * Displays the menu routes as a resource
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class MenuController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        return (new Builder('menus.admin.menus', Menu::query()
                ->root()
                ->buildSearch($request)))
            ->columns([
                    'title' => trans('fields.title'),
                    'status' => trans('fields.status'),
                    'location' => trans('fields.location'),
                    'updated_at' => trans('fields.updated_at')
            ])
            ->data(
                [
                    'title' => function($item) {
                        return $item->title;
                    },
                    'location' => function($item) {
                        return !empty($item->location) ?  $item->location->title : '';
                    },
                    'status' => function($item) {
                        return ucfirst($item->status);
                    }
                ]
            )
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        // Set the default action to create an item
        $action = call_user_func_array(
            'route', [
                "menus.admin.menus.store",
                []
            ]
        );

        $method = 'POST';
        return view('menus::admin.menus.edit')
            ->with([
                'title' => 'Menu',
                'item' => new Menu,
                'node' => new Menu,
                'visible_tree' => [],
                'form_action' => $action,
                'form_method' => $method,
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }

    /**
     * Display the specified resource with it's parent Node
     *
     * @param int $id Menu ID
     *
     * @return \Illuminate\Http\Response JSON response
     */
    public function show(Menu $menu)
    {
        return $menu;
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param StoreMenu $request Validated Request
     * @param Menu      $product Empty product model
     *
     * @return \Illuminate\Http\Response Redirect to edit page
     */
    public function store(StoreMenu $request, Menu $menu)
    {
        $menu->fill($request->input('item', []));
        $menu->fill($request->input('node', []));
        $menu->save();

        session()->flash('success', 'Menu created');
        return redirect(route('menus.admin.menus.edit', [
            'menu' => $menu->id
        ]));
    }

    /**
     * Store the expanded menu items so hierarchy is shown after page reload
     *
     * @param Request $request Request
     * @param Menu    $product Product Model
     *
     * @return null
     */
    public function storeRequest(Request $request, Menu $menu)
    {
        if ($request->input('action') === 'update_visible') {
            $request->session()
                ->put("menu_expanded[$menu->id]", $request->input('visible', []));
        } elseif ($request->input('action') === 'discard') {
            Menu::where('id', $request->input('menu_id'))
                ->delete();
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Menu $product Product Model
     *
     * @return \Illuminate\View\View
     */
    public function edit(Menu $menu)
    {
        // Load children as the pre-loader doesn't do this
        $menu->children;
        // Set the default action to create an item
        $action = call_user_func_array('route', [
            "menus.admin.menus.update",
            [
                'menu' => $menu->id
            ]
        ]);

        // Fetch action returns path without current menu id to allow a different menu id to be passed
        $fetch_action = call_user_func_array('route', [
            "menus.admin.menus.index",
        ]);

        if (session()->has("menu_expanded[{$menu->id}]")) {
            $visible_tree = session("menu_expanded[{$menu->id}]");
        } else {
            $visible_tree = [
                $menu->id
            ];
        }

        // Make sure that root level is available for display
        if (!in_array($menu->id, $visible_tree)) {
            $visible_tree[] = $menu->id;
        }

        $method = 'PUT';
        return view('menus::admin.menus.edit')
            ->with([
                'title' => 'Menu',
                'item' => $menu,
                'node' => $menu,
                'fetch_action' => $fetch_action,
                'form_action' => $action,
                'form_method' => $method,
                'visible_tree' => $visible_tree,
                'sections' => $menu->fetchSections(),
                'route_params' => [ 'menu' => $menu->id],
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreMenu $request Validated Request
     * @param Menu      $product Existing Product model
     *
     * @return \Illuminate\Http\Response Redirect back to product home
     */
    public function update(StoreMenu $request, Menu $menu)
    {
        foreach ($request->input('new', []) as $class => $values) {
            if (class_exists($class)) {
                if (empty($values['id'])) {
                    $parent = Menu::find($request->input('parent_id'));
                    $values['type'] = $class;
                    $parent->children()->create($values);
                } else {
                    $child = Menu::find($values['id']);
                    $child->fill($values);
                    $child->save();
                }
            }
        }

        $menu->fill($request->input('item', []));
        $menu->fill($request->input('node', []));
        $menu->save();

        session()->flash('success', 'Menu updated');
        return redirect(route('menus.admin.menus.edit', [
            'menu' => $menu->id
        ]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request Incoming Request
     * @param Menu $product Product to destroy
     *
     * @return \Illuminate\Http\Response Either JSON success or redirect back
     */
    public function destroy(Request $request, Menu $menu)
    {
        $menu->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "{$menu->title} has been deleted.");
        return redirect()->back();
    }

}
