<?php
/**
 * Menu Class
 *
 * PHP Version 7
 *
 * @category Mtc\Menus
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus;

use Mtc\Core\Node;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Mtc\Menus\Contracts\MenuContract;
use Mtc\Menus\Events\Admin\GetMenuSections;

/**
 * Menu object.
 * Used to define navigation items and hierarchy.
 * Extends Baum\Node to support hierarchy structure
 *
 * @category Mtc\Menus
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Menu extends \Baum\Node implements MenuContract
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'url',
        'title',
        'image',
        'status',
        'allow_update',
        'type'
    ];

    /**
     * Register a custom save event in order to create related node and
     * refresh taxonomy cache.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::saved(
            function ($menu) {
                $root = $menu->getRoot();

                // Clear the root node's cache.
                Cache::forget("menu:{$root->id}");

                // If this menu item was part of a menu, clear cache
                if (!empty($root->location)) {
                    Cache::forget("menu:{$root->location->location_key}:tree");
                }

            }
        );
        static::deleted(
            function ($menu) {
                $root = $menu->getRoot();
                // If this menu item was part of a menu, clear cache
                if (!empty($root->location)) {
                    Cache::forget("menu:{$root->location->location_key}:tree");
                }

            }
        );
    }

    /**
     * Get the related nodes through the node_taxonomy table
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function nodes()
    {
        return $this->belongsToMany(Node::class);
    }

    /**
     * Get the location where this menu is displayed
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function location()
    {
        return $this->hasOne(MenuLocation::class, 'menu_id');
    }

    /**
     * Define the scope for BuildSearch which is required by admin Index listing query
     *
     * @param \Illuminate\Eloquent\Query\Builder $query
     * @param Request $request
     */
    public function scopeBuildSearch($query, Request $request)
    {
    }

    /**
     * Define Scope for root() in query to find the menu instances
     *
     * @param $query
     * @return mixed
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Retrieve list of Menu sections.
     * This is done by firing an event GetMenuSections.
     * Event MUST return an array of sections
     * which is later filtered and merged into a proper response
     *
     * @return array
     */
    public function fetchSections()
    {
        $sections = [];
        $results = array_filter(event(new GetMenuSections($this)));
        foreach ($results as $array) {
            $sections = array_merge($sections, $array);
        }
        return $sections;
    }

    /**
     * Return a URL for the menu item, caching the URL for 30 minutes.
     *
     * @param boolean                $absolute Include the Site's URL.
     * @param boolean|\Mtc\Core\Node $node     The associated node if available
     *
     * @return string
     */
    public function getUrl($absolute = true, $node = false)
    {
        // As this may be called ahead of the node being saved, we need to
        // pass through.
        if ($node instanceof Node === false) {
            $node = $this->node;
        }

        $prefix = '';
        // Set the URL prefix if there is a parent page
        if ($this->parent instanceof Page) {
            $prefix = $this->parent->node->url;
        }

        return Cache::remember(
            "menu:{$this->id}:url",
            30,
            function() use ($absolute, $node, $prefix) {
                if ($absolute) {
                    // Need base url here
                    return request()->root() . $this->url;
                } else {
                    return $this->url;
                }
            }
        );
    }

    /**
     * Retrieve the menu tree with child nodes for a specific location tag
     *
     * @param string $location_tag tag name for location
     * @param int $depth how many levels of menu should be retrieved
     * @return mixed
     */
    public static function loadMenuFromTag($location_tag, $depth = 1)
    {
        return Cache::remember(
            "menu:$location_tag:tree:$depth",
            60,
            function () use ($location_tag, $depth) {
                $with = 'menu.children';
                // If depth is defined, remember child nodes
                if ($depth > 1) {
                    $with .= str_repeat('.children', $depth - 1);
                }
                return MenuLocation::with($with)
                    ->whereHas('menu', function ($query) {
                        $query->where('status', 'published');
                    })
                    ->where('location', $location_tag)
                    ->first();
            }
        );
    }

    /**
     * Merge Menu styling config defaults with the ones passed through template
     *
     * @param string[] $option_array Menu styling options
     * @return string[] Final menu options
     */
    public static function mergeOptions($option_array)
    {
        $defaults = [
            // Define default options here
        ];

        foreach ($option_array as $key => $value) {
            $defaults[$key] = $value;
        }
        return $defaults;
    }
}
