<?php
/**
 * Menu Controller (Admin)
 *
 * PHP Version 7
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus\Http\Controllers\Admin;

use Baum\MoveNotPossibleException;
use Event;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Node;
use Mtc\Menus\Menu;
use Mtc\Menus\Http\Requests\Admin\StoreMenu;

/**
 * Displays the menu routes as a resource
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class MenuController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View|array
     */
    public function index(Request $request)
    {

        if ($request->wantsJson()) {
            $data = Menu::roots()->get();

            return [
                'children' => $data->map(
                    function ($item) {
                        $item->isExpanded = false;
                        return $item;
                    }
                )
            ];
        }

        return (new Builder('menus.admin.menus', Menu::query()
            ->root()
            ->buildSearch($request)))
            ->columns([
                'title' => trans('fields.title'),
                'status' => trans('fields.status'),
                'location' => trans('menus::text.location'),
                'updated_at' => trans('fields.updated_at')
            ])
            ->data(
                [
                    'title' => function ($item) {
                        return $item->title;
                    },
                    'location' => function ($item) {
                        return !empty($item->location) ? $item->location->title : '';
                    },
                    'status' => function ($item) {
                        return ucfirst($item->status);
                    }
                ]
            )
            ->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        // Set the default action to create an item
        $action = route("menus.admin.menus.store");

        $method = 'POST';
        return view('menus::admin.menus.edit')
            ->with([
                'title' => 'Menu',
                'item' => new Menu,
                'node' => new Menu,
                'types' => Menu::getTypes(),
                'form_action' => $action,
                'form_method' => $method,
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }

    /**
     * Display the specified resource with it's parent Node
     *
     * @param int $id Menu ID
     *
     * @return \Illuminate\Http\Response JSON response
     */
    public function show(Menu $menu)
    {
        $menu->children->map(
            function ($child) {
                // Set a default value for Vue to hook into.
                $child->isExpanded = false;
                $child->edit = false;
                return $child;
            }
        );
        return $menu;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreMenu $request Validated Request
     * @param Menu $product Empty product model
     *
     * @return \Illuminate\Http\Response Redirect to edit page
     */
    public function store(StoreMenu $request, Menu $menu)
    {
        $menu->fill($request->input('item', []));
        $menu->fill($request->input('node', []));
        $menu->save();

        session()->flash('success', __('menus::text.created'));
        return redirect(route('menus.admin.menus.edit', [
            'menu' => $menu->id
        ]));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Menu $product Product Model
     *
     * @return \Illuminate\View\View
     */
    public function edit(Menu $menu)
    {
        // Load children as the pre-loader doesn't do this
        $menu->children;
        // Set the default action to create an item
        $action = route("menus.admin.menus.update", [
            'menu' => $menu->id
        ]);

        // Fetch action returns path without current menu id to allow a different menu id to be passed
        $fetch_action = route("menus.admin.menus.index");

        if (session()->has("menu_expanded[{$menu->id}]")) {
            $visible_tree = session("menu_expanded[{$menu->id}]");
        } else {
            $visible_tree = [
                $menu->id
            ];
        }

        // Make sure that root level is available for display
        if (!in_array($menu->id, $visible_tree)) {
            $visible_tree[] = $menu->id;
        }

        $method = 'PUT';
        return view('menus::admin.menus.edit')
            ->with([
                'title' => 'Menu',
                'item' => $menu,
                'node' => $menu,
                'types' => Menu::getTypes(),
                'fetch_action' => $fetch_action,
                'form_action' => $action,
                'form_method' => $method,
                'visible_tree' => $visible_tree,
                'route_params' => ['menu' => $menu->id],
                'hide_description' => true,
                'hide_visibility' => true
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreMenu $request Validated Request
     * @param Menu $product Existing Product model
     *
     * @return \Illuminate\Http\Response Redirect back to product home
     */
    public function update(StoreMenu $request, Menu $menu)
    {
        $menu->fill($request->input('item', []));
        $menu->fill($request->input('node', []));
        $menu->save();

        session()->flash('success', 'Menu updated');
        return redirect(route('menus.admin.menus.edit', [
            'menu' => $menu->id
        ]));
    }

    /**
     * Process the creation / update of a menu entry
     *
     * @param Request $request
     */
    public function process(Request $request)
    {
        $rules = Menu::getValidationRules($request);
        $this->validate($request, $rules);

        $menu_entry = Menu::query()->findOrNew($request->input('id'));
        $menu_entry->fill($request->all());
        $menu_entry->parent_id = $request->input('menu_id');

        /*
         * Check if this menu entry has been set to pre-fill with information from a Node
         * If we fill a node and have selected linking the object we will need to update the data of the menu entry
         */
        if ($menu_entry->node_id > 0 && $menu_entry->lock_edit) {
            $node = Node::query()->find($menu_entry->node_id);
            $menu_entry->title = $node->title;
            $menu_entry->url = $node->url;
        }

        $menu_entry->save();


        // Add additional info for request
        $menu_entry->isExpanded = false;
        $menu_entry->edit = false;
        return $menu_entry;
    }

    /**
     * Remove the specified resource f  rom storage.
     *
     * @param Request $request Incoming Request
     * @param Menu $product Product to destroy
     *
     * @return \Illuminate\Http\Response Either JSON success or redirect back
     */
    public function destroy(Request $request, Menu $menu)
    {
        $menu->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "{$menu->title} has been deleted.");
        return redirect()->back();
    }

    /**
     * List the supported menu types
     *
     * @return \Illuminate\Support\Collection
     */
    public function listMenuTypes()
    {
        return Menu::getTypes();
    }

    /**
     * List menu items as ID => Title pair
     *
     * @param Request $request
     * @return mixed Collection
     */
    public function list(Request $request)
    {
        $menus = Menu::query();

        if ($request->input('exclude', false)) {
            /*
             * To exclude a menu entry we need to find it and all its children
             * This is done by loading a menu entry so we can get its lft & rgt values
             */
            $exclude = Menu::query()->find($request->input('exclude', false));

            $menus->where('id', '!=', $exclude->id)
                ->where(function ($query) use ($exclude) {
                    /*
                     * Make sure that we don't allow entries in range lft..rgt
                     * However we can allow all others ( < lft and > rgt)
                     */
                    $query->where('rgt', '<', $exclude->lft)
                        ->orWhere('lft', '>', $exclude->rgt);
                });
        }

        /*
         * Check if we are restricting the list by depth
         * Location selection only allows root level menu so it needs filtering by depth of 0
         */
        if ($request->input('of_depth', false)) {
            $menus->where('depth', $request->input('of_depth'));
        }

        /*
         * Get entries
         * Ordered by tree hierarchy
         * Fetch only id & title
         */
        $menus = $menus->orderBy('lft')
            ->select([
                'id',
                'title'
            ])
            ->get();

        /*
         * Reorder needs to have key/value map
         */
        if ($request->input('key_value_pair')) {
            $menus = $menus->keyBy('id')
                ->map(function ($menu) {
                    if (is_string($menu)) {
                        return $menu;
                    }

                    return str_repeat('-', $menu->depth) . ' ' . $menu->title;
                });
        }

        if ($request->input('prepend_root')) {
            $menus->prepend(trans('core::fields.root_level'));
        }


        // Return a prepared response
        // Doing this instead of collection as collection is already sorted and we don't want the order to change
        return response()->json($menus);

    }

    /**
     * Update order of given menu list
     *
     * @param Request $request
     */
    public function updateOrder(Request $request)
    {
        /** @var Menu $parent */
        $parent = Menu::find($request->input('menus.0.parent_id'));

        foreach ($request->input('menus', []) as $key => $menu_data) {
            /** @var Menu $menu */
            $menu = Menu::find($menu_data['id']);

            try {
                if (!empty($previous_taxonomy)) {
                    $menu->moveToRightOf($previous_taxonomy);
                } else {
                    $menu->makeFirstChildOf($parent);
                }
            } catch (MoveNotPossibleException $exception) {
                // Ignore impossible move and proceed
                // e.g. the order of the first child has not changed causes the exception
            }

            // set this taxonomy as previous so we can put next one besides this one
            $previous_taxonomy = $menu;
        }
    }

    /**
     * Update order of given taxonomies
     *
     * @param Request $request
     */
    public function changeParent(Request $request)
    {
        /** @var Menu $menu */
        $menu = Menu::find($request->input('menu'));

        if ($request->input('new_parent') == 0) {
            $menu->makeRoot();

        } else {
            $parent = Menu::find($request->input('new_parent'));
            $menu->makeChildOf($parent);
        }
    }

}
