<?php
/**
 * Menu Liocation Controller (Admin)
 *
 * PHP Version 7
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus\Http\Controllers\Admin;

use Event;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Mtc\Menus\Menu;
use Mtc\Menus\MenuLocation;

/**
 * Displays the menu location routes as a resource
 *
 * @category Mtc\Menus\Http\Controllers\Admin
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class LocationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $action = call_user_func_array(
            'route', [
                "menus.admin.locations.store",
            ]
        );

        $menus = Menu::query()
            ->root()
            ->get();

        $method = 'POST';
        return view('menus::admin.menus.locations')
            ->with([
                'title' => 'Menu Locations',
                'item' => new Menu,
                'locations' => MenuLocation::all(),
                'menus' => $menus,
                'form_action' => $action,
                'form_method' => $method,
                'hide_description' => true
            ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request Request
     *
     * @return \Illuminate\Http\Response Redirect to product home
     */
    public function store(Request $request)
    {
        if ($request->input('action') === 'add_new') {
            MenuLocation::create($request->input('new'));
            session()->flash('success', 'Location added');
        } elseif ($request->input('action') === 'update') {
            foreach ($request->input('location', []) as $location_id => $menu_id) {
                $location = MenuLocation::find($location_id);
                $location->menu_id = $menu_id;
                $location->save();
                Cache::forget("menu:$location->location_key:tree");
            }
            session()->flash('success', 'Menu locations updated');
        }
        return back();
    }

    /**
     * List menu locations with their respective menu entries
     *
     * @return \Illuminate\Database\Eloquent\Collection|static[]
     */
    public function list()
    {
        return MenuLocation::query()
            ->with('menu')
            ->get()
            ->map(function ($location) {
                $location->edit = false;
                return $location;
            });
    }

    /**
     * Process the creation / update of a menu location
     *
     * @param Request $request
     */
    public function process(Request $request)
    {
        $this->validate($request, [
            'title' => 'required',
            'location' => 'required|alpha_dash|unique:menu_locations,location,' . $request->input('id'),
            'menu_id' => 'numeric'
        ]);

        $menu_entry = MenuLocation::query()->findOrNew($request->input('id'));
        $menu_entry->fill($request->all());
        $menu_entry->save();

    }

    /**
     * Delete the menu location
     *
     * @param Request $request Incoming Request
     * @param MenuLocation   $group   Custom group
     * @throws \Exception
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, MenuLocation $location)
    {
        $location->delete();

        if ($request->ajax()) {
            return response('success');
        }

        $request->session()->flash('success', "'The Location has been deleted.");
        return redirect()->back();
    }



}
