<?php
/**
 * Menu Class
 *
 * PHP Version 7
 *
 * @category Mtc\Menus
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Mtc\Core\Node;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Spatie\Translatable\HasTranslations;

/**
 * Menu object.
 * Used to define navigation items and hierarchy.
 * Extends Baum\Node to support hierarchy structure
 *
 * @category Mtc\Menus
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Menu extends \Baum\Node
{
    use HasTranslations;
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'url',
        'title',
        'image',
        'status',
        'node_id',
        'css_class',
        'lock_edit',
        'auto_load_children',
        'type'
    ];

    public $translatable = [
        'title',
    ];

    protected $hidden = [
        'created_at',
        'updated_at',
        'lft',
        'rgt'
    ];

    /**
     * Register a custom save event in order to create related node and
     * refresh taxonomy cache.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::saved(
            function ($menu) {
                $root = $menu->getRoot();

                // Clear the root node's cache.
                Cache::forget("menu:{$root->id}");

                // If this menu item was part of a menu, clear cache
                if (!empty($root->location)) {
                    Cache::forget("menu:{$root->location->location_key}:tree");
                }

            }
        );
        static::deleted(
            function ($menu) {
                $root = $menu->getRoot();
                // If this menu item was part of a menu, clear cache
                if (!empty($root->location)) {
                    Cache::forget("menu:{$root->location->location_key}:tree");
                }

            }
        );
    }

    /**
     * Get the related nodes through the node_taxonomy table
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function nodes()
    {
        return $this->belongsToMany(Node::class);
    }

    /**
     * Relationship with only active child entries
     * Used for output on pubic site
     *
     * @return HasMany
     */
    public function activeChildren()
    {
        return $this->children()
            ->where('status', 'published')
            ->orderBy('lft');
    }

    /**
     * Get the location where this menu is displayed
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function location()
    {
        return $this->hasOne(MenuLocation::class, 'menu_id');
    }

    /**
     * Define the scope for BuildSearch which is required by admin Index listing query
     *
     * @param Builder $query
     * @param Request $request
     */
    public function scopeBuildSearch($query, Request $request)
    {
    }

    /**
     * Define Scope for root() in query to find the menu instances
     *
     * @param $query
     * @return mixed
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Return a URL for the menu item, caching the URL for 30 minutes.
     *
     * @param boolean $absolute Include the Site's URL.
     * @param boolean|\Mtc\Core\Node $node The associated node if available
     *
     * @return string
     */
    public function getUrl($absolute = true, $node = false)
    {
        // As this may be called ahead of the node being saved, we need to
        // pass through.
        if ($node instanceof Node === false) {
            $node = $this->node;
        }

        $prefix = '';
        // Set the URL prefix if there is a parent page
        if ($this->parent instanceof Page) {
            $prefix = $this->parent->node->url;
        }

        return Cache::remember(
            "menu:{$this->id}:url",
            30,
            function () use ($absolute, $node, $prefix) {
                if ($absolute) {
                    // Need base url here
                    return request()->root() . $this->url;
                } else {
                    return $this->url;
                }
            }
        );
    }

    /**
     * Retrieve the menu tree with child nodes for a specific location tag
     *
     * @param string $location_tag tag name for location
     * @param int $depth how many levels of menu should be retrieved
     * @return mixed
     */
    public static function loadMenuFromTag($location_tag, $depth = 1)
    {
        return Cache::remember(
            "menu:$location_tag:tree:$depth",
            60,
            function () use ($location_tag, $depth) {
                $with = 'menu.activeChildren';
                // If depth is defined, remember child nodes
                if ($depth > 1) {
                    $with .= str_repeat('.activeChildren', $depth - 1);
                }

                return MenuLocation::with($with)
                    ->whereHas('menu', function ($query) {
                        $query->where('status', 'published');
                    })
                    ->where('location', $location_tag)
                    ->first();
            }
        );
    }

    /**
     * Merge Menu styling config defaults with the ones passed through template
     *
     * @param string[] $option_array Menu styling options
     * @return string[] Final menu options
     */
    public static function mergeOptions($option_array)
    {
        $defaults = [
            // Define default options here
        ];

        foreach ($option_array as $key => $value) {
            $defaults[$key] = $value;
        }
        return $defaults;
    }

    /**
     * Get the supported types of menu entries
     * Return them in a list format used by the menu builder
     *
     * @return Collection
     */
    public static function getTypes()
    {
        /*
         * Define the two basic types of menu entries
         * - Link - a static link field that can be filled in manually
         * - Image - a media file that can be uploaded and linked to a custom url
         */
        $types = collect([
            [
                'id' => self::class,
                'name' => 'Link',
                'fields' => [
                    [
                        'label' => 'URL',
                        'name' => 'url',
                        'type' => 'text',
                        'validation' => 'required',
                    ]
                ]
            ],
            [
                'id' => 'image',
                'name' => 'Image',
                'fields' => [
                    [
                        'label' => 'Image',
                        'name' => 'image',
                        'type' => 'file',
                        'validation' => 'required',
                    ],
                    [
                        'label' => 'URL',
                        'name' => 'url',
                        'type' => 'text',
                        'validation' => 'required',
                    ]
                ]
            ],
        ]);

        Node::query()
            ->where('url', '!=', '')
            ->distinct()
            ->pluck('nodeable_type')
            ->each(function ($entry) use ($types) {
                $types->push([
                    'id' => $entry,
                    'name' => class_basename($entry),
                    'fields' => [
                        [
                            'label' => class_basename($entry),
                            'name' => 'node_id',
                            'type' => 'select',
                            'values' => $entry,
                            'validation' => 'required',
                        ],
                        [
                            'label' => 'Auto-fill information from this entry',
                            'name' => 'lock_edit',
                            'type' => 'checkbox',
                        ],
                        [
                            'label' => '(TODO)Automatically sync child entries',
                            'name' => 'auto_load_children',
                            'type' => 'checkbox',
                        ]
                    ]
                ]);
            });

        return $types;
    }

    /**
     * Get the validation rules for Menu entry submission
     * This checks the and ensures rules are correct for menu entry type
     *
     * @param Request $request
     * @return array
     */
    public static function getValidationRules(Request $request)
    {
        /*
         * Set the basic rules.
         * On all instances we will need the title and type
         * Based on type we will know if any additional validation rules apply
         */
        $rules = collect([
            'title' => 'required',
            'type' => 'required'
        ]);

        /*
         * Fetch all supported types of menu entries
         * Filter them to find the one that has been submitted
         * Add the additional rules from this menu type
         */
        collect(self::getTypes())
            ->filter(function ($type) use ($request) {
                return $type['id'] == $request->input('type');
            })
            ->each (function ($type) use ($rules) {
                /*
                 * Loop through all fields
                 * Ignore the ones that do not have any validation set
                 * Add the validation rules for fields to the collection
                 */
                collect($type['fields'])
                    ->filter(function ($field) {
                        return !empty($field['validation']);
                    })
                    ->each(function ($field) use ($rules) {
                        $rules->put($field['name'], $field['validation']);
                    });

            });

        return $rules->toArray();
    }
}
