<?php
/**
 * Event Service Provider for Menus
 *
 * PHP Version 7
 *
 * @category Mtc\Menus\Providers
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Menus\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use Mtc\Core\Events\Admin\MenuLoading;
use Mtc\Core\Admin\Menu as AdminMenu;
use Mtc\Core\Admin\Tab;
use Mtc\Core\Node;
use Mtc\Menus\Listeners\RemoveMenuEntry;
use Mtc\Menus\Listeners\UpdateMenuEntry;
use Mtc\Menus\Menu;

/**
 * Connect the Menu Events to Laravel
 *
 * @category Mtc\Menus\Providers
 * @package  Mtc\Menus
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class EventServiceProvider extends ServiceProvider
{
    /**
     * Boot the service provider.
     *
     * @return void
     */
    public function boot()
    {
        // fetch the menu on site - called from a template menus::public.menu
        Event::listen(
            'menu.load', function ($location_tag, $depth = 2) {
            return Menu::loadMenufromTag($location_tag, $depth);
        });

        // Admin Navigation Menu construct
        Event::listen(
            MenuLoading::class, function () {
            AdminMenu::addMenu(
                'management',
                __('menus::text.navigation'),
                'manage-navigation',
                'navigation',
                'glyphicon glyphicon-list');

            AdminMenu::addSubMenu(
                'navigation',
                __('menus::text.navigation_menus'),
                '',
                route('menus.admin.menus.index'),
                'navigation');

            AdminMenu::addSubMenu(
                'navigation',
                __('menus::text.menu_locations'),
                '',
                route('menus.admin.locations.index'),
                'navigation');
        }
        );

        /*
         * Add Tabs for navigation on Menu Management section
         * This event adds navigation tabs for menu management sections
         * when in admin manage menu or manage menu location
         */
        // Edit page - add tabs for navigation
        Event::listen('tabs.loading: ' . Menu::class,
            function ($model) {
                /*
                 * We need to check if we are able to link the menu instance
                 * to a menu entry. If we have model we can add this model entry
                 * in tabs. Alternatively we skip this menu entry
                 */
                $model_menu = [];
                if (!empty($model->id)) {
                    $model_menu = new Tab(
                        'menus.index',
                        $model->title,
                        route('menus.admin.menus.edit', $model)
                    );
                }

                /*
                 * Return a list of tabs
                 * - manage page
                 * - edit menu (if menu model available)
                 * - manage locations
                 */
                return [
                    new Tab(
                        'menus.index',
                        __('menus::text.manage_menus'),
                        route('menus.admin.menus.index')
                    ),
                    $model_menu,
                    new Tab(
                        'menus.locations',
                        __('menus::text.manage_locations'),
                        route('menus.admin.locations.index')
                    ),
                ];
            }
        );

        /*
         * We need to make sure we update menu entries when models are changed
         * This is for locked / linked menu entries only
         * We listen to model saving & deleting events to make sure
         * that we can check any changes against menu entries and update menu entries
         * that are linked and locked to this model.
         */
        Event::listen('eloquent.saving: ' . Node::class, UpdateMenuEntry::class);
        Event::listen('eloquent.deleting: ' . Node::class, RemoveMenuEntry::class);
    }
}
