<?php
/**
 * Newsletter List Controller
 *
 * This controller is used for managing newsletter lists on site
 *
 * @category Mtc\Components\Newsletter\Http\Controllers\Admin
 * @package  Mtc\Components\Newsletter\Http\Controllers
 * @author Justyna Cala <justyna.cala@mtcmedia.co.uk>
 * @version 2017.06.13
 */

namespace Mtc\Components\Newsletter\Http\Controllers\Admin;

use Carbon\Carbon;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Components\Newsletter\NewsletterList;
use Mtc\Components\Newsletter\NewsletterEmail;
use League\Csv\Writer;
use Event;

/**
 * Class NewsletterController
 *
 * @package Mtc\Components\Newsletter\Http\Controllers\Admin
 */
class NewsletterController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {

        return (new Builder('newsletter.admin.lists', NewsletterList::query(), 'newsletter::titles.lists'))
            ->columns(
                [
                    'name' => trans('fields.name'),
                    'emails_no' => trans('newsletter::fields.email_no'),
                    'is_default' => trans('newsletter::fields.is_default'),
                ])
            ->data([
                'emails_no' => function ($list) {
                    return $list->emails->count();
                },
                'is_default' => function ($list) {
                    return $list->is_default ? trans('newsletter::fields.yes') : trans('newsletter::fields.no');
                }
            ])
            ->view();


    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param NewsletterList $list model
     *
     * @return \Illuminate\View\View
     */
    public function edit(NewsletterList $list)
    {

        $resource_name = 'newsletter.admin.lists';

        // Action for updating a newsletter list
        $action = route(...[
            "{$resource_name}.update",
            [$list->id]
        ]);

        return view('core::admin.builder.item')
            ->with([
                'title' => 'newsletter::titles.list',
                'view' => 'newsletter::admin.newsletter.form',
                'name' => $resource_name,
                'item' => $list,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request Request with validation
     * @param NewsletterList $list model
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, NewsletterList $list)
    {
        return $this->store($request, $list);
    }

    /**
     * Create or update NewsletterList
     *
     * @param Request $request Request
     * @param NewsletterList $list model
     *
     * @return \Illuminate\Http\Response Redirect to default listing
     */
    public function store(Request $request, NewsletterList $list)
    {
        $this->validate($request, [
            'name' => 'required',
        ]);

        $add = empty($list->id);
        $list->fill($request->all());
        $list->is_default = (int)$request->has('is_default');

        //remove previous default before saving
        if (!empty($list->is_default)) {
            self::removePreviousDefault();
        }

        $list->save();

        if ($add) {
            $request->session()->flash('success', trans('newsletter::titles.newsletter_added') . $list->name);
        } else {
            $request->session()->flash('success', trans('newsletter::titles.newsletter_updated') . $list->name);
        }
        return redirect(route('newsletter.admin.lists.index'));
    }

    /**
     * Export NewsletterList emails
     *
     * @param NewsletterList $list
     *
     * @return void
     */
    public function exportAndDie(NewsletterList $list)
    {
        $emails = $list->emails->toArray();

        $csv = Writer::createFromFileObject(new \SplTempFileObject());
        // Add Titles
        $csv->insertOne([
            'Email',
            'Date'
        ]);

        foreach ($emails as $email_sign_up) {
            // Add Data
            $csv->insertOne([$email_sign_up['email'], $email_sign_up['created_at']]);
        }

        $csv->output($list->name . "_emails_" . date('Y-m-d_H-i-s') . ".csv");
        exit;
    }

    /**
     * Show the page to create a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $list = new NewsletterList;
        $resource_name = 'newsletter.admin.lists';
        $action = route(...[
            "{$resource_name}.store",
            [$list->id]
        ]);

        return view('core::admin.builder.item')
            ->with([
                'title' => 'Newsletter Lists',
                'view' => 'newsletter::admin.newsletter.form',
                'name' => $resource_name,
                'item' => $list,
                'node' => null,
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param NewsletterList $list
     *
     * @return \Illuminate\Http\Response Redirect back
     */
    public function destroy(Request $request, NewsletterList $list)
    {
        $list->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', trans('newsletter::titles.newsletter_deleted') . $list->name);
        return redirect()->back();
    }

    /**
     * Remove previous default
     *
     */
    public static function removePreviousDefault()
    {
        //deselect default from previous list if available
        NewsletterList::where('is_default', 1)
            ->update([
                'is_default' => 0
            ]);

    }
}