<?php
/**
 * Newsletter Tests
 *
 * @category Mtc\Components\Newsletter\Tests
 * @package Mtc\Components\Newsletter
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

use Illuminate\Foundation\Testing\DatabaseTransactions;
use Mtc\Components\Newsletter\Http\Controllers\NewsletterController;
use Mtc\Components\Newsletter\Http\Controllers\Admin\NewsletterController as AdminNewsletterController;
use Mtc\Components\Newsletter\NewsletterList;

/**
 * Newsletter Tests
 *
 * These are the tests for confirming the newsletter
 * functionality is working without any unexpected issues
 *
 * @category Mtc\Components\Newsletter\Tests
 * @package Mtc\Components\Newsletter
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class NewsletterTest extends \Tests\TestCase
{
    use DatabaseTransactions;

    /**
     * Test that sign up page loads successfully
     */
    public function testSignUpPageLoads()
    {
        $this->get(action('\\' . NewsletterController::class . '@showSignUpForm'))
            ->assertStatus(200);
    }

    /**
     * Test that signed up page loads successfully
     */
    public function testSignedUpPageLoads()
    {
        $this->get(action('\\' . NewsletterController::class . '@checkIfSignedUpSession'))
            ->assertStatus(200);
    }

    /**
     * Test that a valid sign-up request results in success message
     */
    public function testValidSignUpRequest()
    {
        // We confirm that 200 status is sent
        // And also confirm that the JSON result is returned as it should be
        $this->post(action('\\' . NewsletterController::class . '@signUp'), [
            'newsletter_email' => str_random(15) . '@example.com'
        ])->assertStatus(200)
            ->assertJson([
                'result' => 'success'
            ]);
    }

    /**
     * Test that invalid request results in failed sign-up
     */
    public function testInvalidSignUpRequest()
    {
        // If email isn't valid a 302 redirect is returned by default to go back to form and allow editing it
        $this->post(action('\\' . NewsletterController::class . '@signUp'), [
            'newsletter_email' => 'invalid-email'
        ])->assertStatus(302);
    }

    /**
     * Test that sign-up list page is not crashing
     */
    public function testViewSignUpLists()
    {
        // attempt to load the admin page for listing the newsletter lists as mtc user
        $this->actingAs(\Mtc\Core\Auth\User::role('mtc')->first())
            ->get(action('\\' . AdminNewsletterController::class . '@index'))
            ->assertStatus(200);
    }

    /**
     * Test that create sign-up list page is not crashing
     */
    public function testCreateSignUpList()
    {
        // attempt to load the admin page for listing the newsletter lists as mtc user
        $this->actingAs(\Mtc\Core\Auth\User::role('mtc')->first())
            ->get(action('\\' . AdminNewsletterController::class . '@create'))
            ->assertStatus(200);
    }

    /**
     * Test that edit sign-up list page is not crashing
     */
    public function testEditSignUpList()
    {
        // attempt to load the admin page for listing the newsletter lists as mtc user
        $this->actingAs(\Mtc\Core\Auth\User::role('mtc')->first())
            ->get(action('\\' . AdminNewsletterController::class . '@edit', [
                NewsletterList::first()->id ?? 0
            ]))
            ->assertStatus(200);
    }

    /**
     * Test that the export functionality is not breaking
     */
    public function testAbleToExportNewsletterList()
    {
        $newsletter_list = NewsletterList::first();
        $content = AdminNewsletterController::export($newsletter_list);
        $this->assertInstanceOf(\League\Csv\Writer::class, $content);
    }
}
