<?php
/**
 * PayPal Payment (Button) Service Provider
 *
 * PHP Version 7
 *
 * @category Mtc\Components\Paypal\Providers
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Components\Paypal\Providers;

use Event;
use Mtc\Components\Paypal\Console\Commands\MakePaypalCertificate;
use Illuminate\Support\ServiceProvider;
use Mtc\Components\Paypal\Http\Controllers\PaypalController;
use Mtc\Components\Paypal\Paypal;
use Mtc\Shop\Events\RetrievePaymentGateways;
use Route;

/**
 * Connect the PayPal Payment Provider to Laravel
 *
 * @category Mtc\Components\Paypal\Providers
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PaypalServiceProvider extends ServiceProvider
{
    /**
     * Boot the service provider.
     *
     * @return void
     */
    public function boot()
    {
        $this->defineResources();
        $this->defineEvents();
        $this->defineRoutes();
        $this->definePublished();
        $this->defineCommands();
    }

    /**
     * Define the command line actions that are set up in this component
     *
     * @return void
     */
    protected function defineCommands()
    {
        $this->commands([
            MakePaypalCertificate::class
        ]);
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {

        $this->mergeConfigFrom(
            __DIR__ . '/../../config/paypal.php', 'paypal'
        );
    }

    /**
     * Define the resources for this package.
     *
     * @return void
     */
    protected function defineResources()
    {
        $this->loadViewsFrom(__DIR__ . '/../../resources/views', 'paypal');

        $this->publishes(
            [
            __DIR__.'/../../config/paypal.php' => config_path('paypal.php'),
            ]
        );
    }

    /**
     * Load any events required for this package.
     *
     * This listens to see if the 'RetrievePaymentGateways' event is triggered
     * and returns a new copy of the form if so.
     *
     * @return void
     */
    protected function defineEvents()
    {
        Event::listen(
            RetrievePaymentGateways::class, function () {
                return new Paypal();
            }
        );
    }

    /**
     * Load routes to send the form and to process the incoming request.
     *
     * @return void
     */
    protected function defineRoutes()
    {
        if (!$this->app->routesAreCached()) {

            Route::group(
                ['middleware' => 'web'], function ($router) {
                    $router->get('payment/paypal', PaypalController::class . '@index')
                        ->name('paypal.index');

                    $router->post('payment/paypal/process', PaypalController::class . '@process')
                        ->name('paypal.process');

                    $router->get('payment/paypal/success', PaypalController::class . '@success')
                        ->name('paypal.success');

                    $router->get('payment/paypal/failed', PaypalController::class . '@failed')
                        ->name('paypal.failed');
                }
            );

        };
    }

    /**
     * Copy development certificates to resource dir within app
     *
     * @return void
     */
    protected function definePublished()
    {
        $this->publishes([
            __DIR__ . '/../../resources/certificates' => resource_path('certificates')
        ], 'certificates');

    }
}
