<?php
/**
 * Controller for PayPal button Payment integration
 *
 * PHP Version 7
 *
 * @category Mtc\Components\Paypal\Http\Controllers
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Components\Paypal\Http\Controllers;

use Config;
use Illuminate\Http\Request;
use Mtc\Components\Paypal\Paypal;
use Mtc\Components\Paypal\PaypalButton;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Shop\Basket;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Order\OrderNote;
use Session;

/**
 * Front-end controls for handling payments with Sagepay Form's payment
 * gateway.
 *
 * @category Mtc\Components\Paypal\Http\Controllers
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PaypalController extends Controller
{
    /**
     * Load the form which will redirect the user to PayPal.
     *
     * @param Request       $request Incoming Request
     * @param OrderContract $order   Order to be paid for
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request, OrderContract $order)
    {
        // Get the current order. If doesn't exist, redirect back to overview.
        $order_id = $request->session()->get('order_id');
        $order = $order->find($order_id);
        if (empty($order)) {
            return redirect(route('shop.basket.overview'));
        }

        // Get the vendor and create a unique vendor code for Sagepay.
        $vendor = Config::get('paypal.vendor_name');
        $vendor_code = $vendor . '-' . date('YmdHis') . '-' . $order_id;
        $request->session()->put('vendor_code', $vendor_code);

        // Retrieve the address and environment details.
        $environment = strtolower(Config::get('paypal.environment'));
        $billing = $order->addresses->where('type', 'billing')->first();
        $shipping = $order->surcharges
            ->where('display_name', ucwords(Basket::SHIPPING_METHOD_SURCHARGE_TYPE))
            ->first();

        $crypt = [
            'cmd' => '_xclick',
            'item_name' => Config::get('app.name') . ' Merchandise',
            'business' => Config::get('paypal.email'),
            'invoice' => $vendor_code,
            'amount' => ($order->total - $shipping->value) / 100,
            'tax' => 0, // TODO: add Tax to Order
            'shipping' => $shipping->value / 100,
            'currency_code' => 'GBP',
            'lc' => 'GB',
            'bn' => 'PP-BuyNowBF',
            'address_override' => 0,
            'first_name' => $billing->first_name,
            'last_name' => $billing->last_name,
            'address1' => $billing->address1,
            'city' => $billing->city,
            'state' => $billing->country == 'US' ? $billing->state : '',
            'zip' => $billing->postcode,
            'country' => $billing->country,
            'notify_url' => route('paypal.process'),
            'return' => route('paypal.success'),
            'return_cancel' => route('paypal.failed'),
            'rm' => 1, // User returns via GET request without any params
        ];

        $button = new PaypalButton($environment);
        $params = [
            'cmd' => '_s-xclick',
            'encrypted' => $button->encrypt($crypt)
        ];

        // Determine the URL we will send the user to, depending on the env.
        $action = ($environment === 'live') ? Config::get('paypal.live_url') : Config::get('paypal.sandbox_url');

        return view('paypal::form')
            ->with(compact('params', 'environment', 'action'));
    }

    /**
     * Process a return from paypal after successful payment.
     *
     * @param Request       $request Incoming Request
     * @param OrderContract $order   Order to be paid for
     * @param OrderNote     $note    New note object to save status
     *
     * @return \Illuminate\View\View  Success view for payment.
     */
    public function success(Request $request, OrderContract $order, OrderNote $note)
    {
        // Get the current order. If doesn't exist, redirect back to overview.
        $order_id = $request->session()->get('order_id');
        $order = $order->find($order_id);
        if (empty($order)) {
            return redirect(route('paypal.failed'));
        }

        // Paypal uses IPN so markPaid won't take care of removing basket ID
        session()->forget('basket_id');
        return view('shop::public.order.success', compact('order'));

    }

    /**
     * Process a cancelled payment page.
     *
     * @param Request       $request Incoming Request
     * @param OrderContract $order   Order to be paid for
     * @param OrderNote     $note    New note object to save status
     *
     * @return \Illuminate\View\View  Fail view for payment.
     */
    public function failed(Request $request, OrderContract $order, OrderNote $note)
    {
        // Get the current order. If doesn't exist, redirect back to overview.
        $order_id = $request->session()->get('order_id');
        $order = $order->find($order_id);
        if (empty($order)) {
            return redirect(route('paypal.failed'));
        }

        return view('shop::public.order.failed', compact('order'));

    }

    /**
     * Process PayPal IPN request
     * This is a background POST request so no requirements to render any template afterwards
     *
     * @param Request $request
     * @param OrderContract $order
     * @param OrderNote $note
     */
    public function process(Request $request, OrderContract $order, OrderNote $note)
    {
        // Verify request to ensure this is not a spoofed order
        $gateway = new Paypal();
        $verified = $gateway->verify($request->input());

        // Locate the order
        $order_id = explode('-', $request->input('invoice'));
        $order_id = end($order_id);
        $order = $order->find($order_id);

        // No order found
        if (empty($order->id)) {
            return;
        }

        if (!$verified) {
            $note->message = "Paypal transaction failed. The Payment vas not verified";
            $order->notes()->save($note);
            return;
        }

        if (in_array($request->input('payment_status'), ['Completed', 'In-Progress', 'Processed', 'Pending'])) {
            $order->payment = $request->input();
            $order->save();

            // Let us know that the order was paid successfully.
            if (false == $order->is_paid) {
                $note->message = "Paypal payment completed. The reference number is " . $request->input('txn_id');
                $order->notes()->save($note);
                $order->markPaid();
            }
        } else {
            $note->message = "Paypal transaction failed. The Payment status was  " . $request->input('payment_status');
            $order->notes()->save($note);
        }

    }

}
