<?php
/**
 * PayPal Payment Gateway
 *
 * PHP Version 7
 *
 * @category Mtc\Components\Paypal
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Components\Paypal;

use Config;
use Mtc\Shop\Abstracts\PaymentGateway;

/**
 * PayPal Payment Gateway
 *
 * @category Mtc\Components\Paypal
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Paypal extends PaymentGateway
{
    /**
     * Payment method title for the frontend.
     *
     * @var string
     */
    public $title = 'PayPal';

    /**
     * The vendor password, defined within the config.
     *
     * @var string
     */
    protected $vendor_pass = '';

    /**
     * Retrieve the vendor password from the Config.
     *
     * @return void
     */
    public function __construct()
    {
        $this->vendor_pass = Config::get('paypal.encryption_password');
        parent::__construct();
    }

    /**
     * Get a link to the transaction on the 3rd party gateway size (if applicable).
     *
     * @param Order $order The order object.
     *
     * @return string        Transaction URL, or empty string.
     */
    public static function getTransactionUrl($order)
    {
        return route('paypal.index');
    }

    /**
     * Verify if the process request is a valid request or it is being faked.
     * This is done by making a request to paypal to check whether order was indeed placed
     *
     * @param string[] $request
     * @return bool
     */
    public function verify($params): bool
    {
        // build callback request
        $request = 'cmd=_notify-validate';
        foreach($params as $key => $value) {
            $request .= "&" . $key . "=" . urlencode($value);
        }

        $port = 443;
        $protocol = 'ssl://';
        $gateway = Config::get('paypal.environment') === 'live' ? 'www.paypal.com' : 'www.sandbox.paypal.com';

        // build callback headers
        $header = "POST /cgi-bin/webscr HTTP/1.1\r\n";
        $header .= "Content-Type: application/x-www-form-urlencoded\r\n";
        $header .= "Content-Length: " . strlen($request) . "\r\n";
        $header .= "Host: " . $gateway . "\r\n";
        $header .= "Connection: close\r\n";
        $header .= "\r\n";

        // Send the verify request
        $fp = fsockopen($protocol . $gateway, $port, $err_no, $err_str, 30);
        fwrite($fp, $header . $request);

        // Get the result back
        $result = '';
        while(!feof($fp)) {
            $result .= fgets($fp, 1024);
        }
        fclose($fp);

        // Check if the returned string has Verified in it
        return strpos($result, 'VERIFIED') !== false;
    }

}
