<?php
/**
 * PayPal Payment Button encryption class
 *
 * PHP Version 7
 *
 * @category Mtc\Components\Paypal
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Components\Paypal;

use \Exception;

/**
 * PayPal Payment Button encryption class
 * Stores the request encryption functionality to ensure Paypal
 * requests cannot be broken by adjusting the hidden input values.
 *
 * @category Mtc\Components\Paypal
 * @package  Mtc\Components\Paypal
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class PaypalButton
{

    /** @var string $certificate Certificate resource */
    protected $certificate;
    /** @var string $certificate_file Path to the certificate file */
    protected $certificate_file;

    /** @var string $private_key Private key resource (matching certificate) */
    protected $private_key;
    /** @var string $private_key_file Path to the private key file */
    protected $private_key_file;
    /** @var string $paypal_certificate PayPal public certificate resource */
    protected $paypal_certificate;
    /** @var string $paypal_certificate_file Path to PayPal public certificate file */
    protected $paypal_certificate_file;
    /** @var string $certificate_id ID assigned by PayPal to the $certificate. */
    protected $certificate_id;
    /** @var string $temp_file_directory */
    protected $temp_file_directory;

    /**
     * Constructor.
     * Creates the object ands sets certifictates based on environment
     *
     * @param string $environment
     */
    public function __construct($environment = 'test')
    {
        // Set the prefix for either live or dev certificate
        $prefix = $environment === 'live' ? '/' : '/dev_';

        // Define file paths for both our an paypal certificate and our private key
        $certificate_file = resource_path('certificates') . $prefix . 'ours_paypal.crt';
        $private_key_file = resource_path('certificates') . $prefix . 'ours_paypal.key';
        $paypal_certificate = resource_path('certificates') . $prefix . 'paypal.crt';

        // set the certificate ID
        $certificate_id = env('paypal_cert_id', 'BFNYE9SV627VL');
        $this->setTempFileDirectory('/tmp');
        $this->setOurCertificate($certificate_file, $private_key_file, $certificate_id);
        $this->setPayPalCertificate($paypal_certificate);
    }

    /**
     * setTempFileDirectory: Sets the directory into which temporary files are written.
     *
     * @param string $directory - Directory in which to write temporary files.
     *
     * @return $this
     * @throws Exception
     */
    protected function setTempFileDirectory($directory)
    {
        if (is_dir($directory) && is_writable($directory)) {
            $this->temp_file_directory = $directory;
            return $this;
        } else {
            throw new Exception('PayPal button temporary directory not accessible!');
        }
    }


    /**
     * setCertificate: set the client certificate and private key pair.
     *
     * @param string $certificate_filename - The path to the client certificate
     * @param string $private_key_filename - The path to the private key corresponding to the certificate
     *
     * @return $this
     * @throws Exception
     */
    protected function setOurCertificate($certificate_filename, $private_key_filename, $certificate_id)
    {

        if (is_readable($certificate_filename) && is_readable($private_key_filename)) {
            $certificate = openssl_x509_read(file_get_contents($certificate_filename));
            $private_key = openssl_get_privatekey(file_get_contents($private_key_filename));

            $cert_and_key_valid = ($certificate !== false) && ($private_key !== false);
            if ($cert_and_key_valid && openssl_x509_check_private_key($certificate, $private_key)) {
                $this->certificate = $certificate;
                $this->certificate_file = $certificate_filename;

                $this->private_key = $private_key;
                $this->private_key_file = $private_key_filename;

                $this->certificate_id = $certificate_id;
                return $this;
            }
        }

        throw new Exception('PayPal button certificate and key don\'t match!');
    }


    /**
     * setPayPalCertificate: Sets the PayPal certificate
     *
     * @param string $file_name - The path to the PayPal certificate.
     *
     * @return $this
     * @throws Exception
     */
    public function setPayPalCertificate($file_name)
    {
        if (is_readable($file_name)) {
            $certificate = openssl_x509_read(file_get_contents($file_name));

            if ($certificate !== false) {
                $this->paypal_certificate = $certificate;
                $this->paypal_certificate_file = $file_name;

                return $this;
            }
        }

        throw new Exception('Could not read PayPal button certificate!');
    }

    /**
     * encrypt: Using the previously set certificates and tempFileDirectory
     * encrypt the button information.
     *
     * @param array $parameters - Array with parameter names as keys.
     *
     * @return string The encrypted string for the _s_xclick button form field.
     * @throws Exception
     */
    public function encrypt($parameters)
    {
        // Check encryption data is available.
        if (!$this->isReady()) {
            throw new Exception('PayPal button encryption not ready!');
        }

        // Compose plain text data.
        $parameters['cert_id'] = $this->certificate_id;
        $plain_text = $this->makeString($parameters);

        $clear_file = tempnam($this->temp_file_directory, 'clear_');
        $signed_file = str_replace('clear', 'signed', $clear_file);
        $encrypted_file = str_replace('clear', 'encrypted', $clear_file);

        $out = fopen($clear_file, 'wb');
        fwrite($out, $plain_text);
        fclose($out);

        if (!openssl_pkcs7_sign(
            $clear_file,
            $signed_file,
            $this->certificate,
            $this->private_key,
            [],
            PKCS7_BINARY
        )) {
            throw new Exception('PayPal button could not be signed with provided credentials!');
        }

        $signed_data = explode("\n\n", file_get_contents($signed_file));

        $out = fopen($signed_file, 'wb');
        fwrite($out, base64_decode($signed_data[1]));
        fclose($out);

        if (!openssl_pkcs7_encrypt(
            $signed_file,
            $encrypted_file,
            $this->paypal_certificate,
            [],
            PKCS7_BINARY
        )) {
            throw new Exception('PayPal button could not be encrypted with provided credentials!');
        }

        $encryptedData = explode("\n\n", file_get_contents($encrypted_file));
        $cipher_text = $encryptedData[1];

        @unlink($clear_file);
        @unlink($signed_file);
        @unlink($encrypted_file);

        return $this->wrapInPkcsHeader($cipher_text);
    }

    /**
     * Wrap the encrypted text in a header and strip out new-lines
     *
     * @param string $cipher_text
     * @return string
     */
    protected function wrapInPkcsHeader($cipher_text)
    {
        return '-----BEGIN PKCS7-----' . str_replace("\n", "", $cipher_text) . '-----END PKCS7-----';
    }

    /**
     * Check if certificates are set for this object
     * Used as a fail-safe to prevent broken integration
     *
     * @return bool
     */
    protected function isReady()
    {
        return ($this->certificate_id != '')
            && isset($this->certificate)
            && isset($this->paypal_certificate);
    }

    /**
     * Convert the param array to a string.
     * This is needed for encrypting the request with certificate\
     *
     * @param string[] $params
     * @return string
     */
    protected function makeString($params)
    {
        $string = '';

        foreach ($params as $key => $value) {
            $string .= empty($value) ? '' : ($key . '=' . $value . "\n");
        }

        return $string;
    }
}