<?php
/**
 * PhoneOrder Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Components\PhoneOrder\Controllers
 * @author   : Joe Pritchard <joe.pritchard@mtcmedia.co.uk>
 *
 * Created: 21/03/2017 11:43
 */
namespace Mtc\Components\PhoneOrder\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Mtc\Components\PhoneOrder\PhoneOrder;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Events\OrderPaid;
use Mtc\Shop\Order\OrderNote;

/**
 * Class PhoneOrderController
 *
 * @package Mtc\Components\PhoneOrder
 */
class PhoneOrderController extends Controller
{
    /**
     * Ensure phone orders can only be placed by those with the permission to do so.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:create-phone-orders');
    }

    /**
     * Display the form to create a phone order.
     *
     * @param Request       $request The current Http request
     * @param OrderContract $order   The current order object
     *
     * @return View
     */
    public function create(Request $request, OrderContract $order)
    {
        $order = $order->find($request->session()->get('order_id'));
        if (empty($order)) {
            return redirect(route('shop.basket.overview'));
        }

        $gateway = new PhoneOrder();
        $action  = $gateway->getReturnUrl($order);

        return view('phone-orders::form')
            ->withAction($action)
            ->withOrder($order)
            ->withGateway($gateway);
    }

    /**
     * Store a newly created phone order.
     *
     * @param  \Illuminate\Http\Request $request
     * @param OrderContract             $order
     * @param OrderNote                 $note
     *
     * @return View
     */
    public function store(Request $request, OrderContract $order, OrderNote $note)
    {
        $this->validate($request, [
            'order_id'       => 'required|numeric',
            'transaction_id' => 'required|alpha_dash',
        ]);

        $gateway        = new PhoneOrder();
        $order_id       = $request->get('order_id');
        $transaction_id = $request->get('transaction_id');

        $order               = $order::find($order_id);
        $order->payment      = [
            'recorded by'    => Auth::user()->name,
            'type'           => $gateway->title,
            'transaction_id' => $transaction_id,
            'amount'         => $order->total / 100,
        ];
        $order->payment_type = $gateway->title;
        $order->save();

        $note->message = "Phone payment recorded, with transaction ID " . $transaction_id;
        $order->notes()->save($note);
        $order->markPaid();

        return view('shop::public.order.success');
    }
}
