<?php
/**
 * Controller for Sagepay Form
 *
 * PHP Version 7
 *
 * @category Mtc\Components\SagepayForm\Http\Controllers
 * @package  Mtc\Components\SagepayForm
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Components\SagepayForm\Http\Controllers;

use Mtc\Core\Http\Controllers\Controller;
use Config;
use Illuminate\Http\Request;
use Mtc\Components\SagepayForm\SagepayForm;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Order\OrderNote;
use Session;

/**
 * Front-end controls for handling payments with Sagepay Form's payment
 * gateway.
 *
 * @category Mtc\Components\SagepayForm\Http\Controllers
 * @package  Mtc\Components\SagepayForm
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class SagepayFormController extends Controller
{
    /**
     * Load the form which will redirect the user to Sagepay.
     *
     * @param Request       $request Incoming Request
     * @param OrderContract $order   Order to be paid for
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request, OrderContract $order)
    {
        // Get the current order. If doesn't exist, redirect back to overview.
        $order_id = $request->session()->get('order_id');
        $order = $order->find($order_id);
        if (empty($order)) {
            return redirect(route('shop.basket.overview'));
        }

        $gateway = new SagepayForm;

        // Get the vendor and create a unique vendor code for Sagepay.
        $vendor = Config::get('sagepay-form.vendor_name');
        $vendor_code = $vendor . '-' . date('YmdHis') . '-' . $order_id;
        $request->session()->put('vendor_code', $vendor_code);

        // Retrieve the address and environment details.
        $environment = strtolower(Config::get('sagepay-form.environment'));
        $billing = $order->addresses->where('type', 'billing')->first();
        $shipping = $order->addresses->where('type', 'shipping')->first() ?: $billing;

        // Build up the encrypted form.
        $crypt = [
            'VendorTxCode' => $vendor_code,
            'Amount' => round($order->total / 100, 2),
            'Currency' => 'GBP',
            'Description' => Config::get('app.name') . ' Merchandise',
            'SuccessURL' => route('sagepay-form.process'),
            'FailureURL' => route('sagepay-form.process'),
            'CustomerName' => $billing->name,
            'CustomerEMail' => $order->email,
            'VendorEMail' => Config::get('sagepay-form.vendor_email'),
            'SendEMail' => Config::get('sagepay-form.send_email'),
            'EmailMessage' => Config::get('sagepay-form.email_message'),
            'BillingSurname' => $billing->last_name,
            'BillingFirstnames' => $billing->first_name,
            'BillingAddress1' => $billing->address1,
            'BillingAddress2' => $billing->address2,
            'BillingCity' => $billing->city,
            'BillingPostCode' => $billing->postcode,
            'BillingCountry' => $billing->country,
            'BillingState' => $billing->country == 'US' ? $billing->state : '',
            'BillingPhone' => $order->phone,
            'DeliverySurname' => $shipping->last_name,
            'DeliveryFirstnames' => $shipping->first_name,
            'DeliveryAddress1' => $shipping->address1,
            'DeliveryAddress2' => $shipping->address2,
            'DeliveryCity' => $shipping->city,
            'DeliveryPostCode' => $shipping->postcode,
            'DeliveryCountry' => $shipping->country,
            'DeliveryState' => $shipping->country == 'US' ? $shipping->state : '',
            'DeliveryPhone' => $order->phone,
            'ReferrerID' => '968FB4B0-E929-45E4-A5BC-41A4FFE873F2',
            'Website' => $request->root(),
        ];

        if ($environment !== 'live') {
            // Disable the default surcharge on the test account
            $crypt['SurchargeXML'] = '<surcharges><surcharge><paymentType>VISA</paymentType><percentage>0</percentage></surcharge></surcharges>';
        }

        // Build the encryption string. Sagepay does not like http_build_query()!
        $post_values = collect($crypt)
            ->map(
                function ($value, $key) {
                    return "{$key}=" . trim($value);
                }
            )->implode('&');

        // Build for unencrypted form params
        $params = [
            'VPSProtocol' => '3.00',
            'TxType' => 'PAYMENT',
            'Vendor' => Config::get('sagepay-form.vendor_name'),
            'Crypt' => $gateway->encryptAndEncode($post_values),
        ];

        // Determine the URL we will send the user to, depending on the env.
        $action = 'https://test.sagepay.com/gateway/service/vspform-register.vsp';
        if ($environment == 'live') {
            $action = 'https://live.sagepay.com/gateway/service/vspform-register.vsp';
        }

        return view('sagepay-form::form')
            ->with(compact('params', 'environment', 'action'));
    }

    /**
     * On return from Sagepay, process the returned and encrypted string to
     * mark the order as paid.
     *
     * @param Request       $request Incoming Request
     * @param OrderContract $order   Order to be paid for
     * @param OrderNote     $note    New note object to save status
     *
     * @return \Illuminate\View\View  Success/Fail view depending on outcome.
     */
    public function process(Request $request, OrderContract $order, OrderNote $note)
    {
        $gateway = new SagepayForm;
        $response = $gateway->decode(str_replace(' ', '+', $request->input('crypt')));

        // Locate the order
        $order_id = explode('-', $response['VendorTxCode']);
        $order_id = end($order_id);
        $order = $order->find($order_id);

        // Determine a default view (will be overridden if failure)
        $view = 'shop::public.order.success';

        if (in_array($response['Status'], ['OK', 'AUTHENTICATED', 'REGISTERED'])) {
            $order->payment = $response;
            $order->save();

            // Let us know that the order was paid successfully.
            if (false == $order->is_paid) {
                $note->message = "Sagepay Form payment completed. The reference number is " . $response['VPSTxId'];
                $order->notes()->save($note);
                $order->markPaid();
            }
        } else {
            $note->message = "Sagepay Form transaction failed. The Error Message was " . $response['StatusDetail'];
            $order->notes()->save($note);
            $view = 'shop::public.order.failed';
        }

        return view($view, compact('order'));
    }

}
