<?php
/**
 * Sagepay Form Payment Gateway
 *
 * PHP Version 7
 *
 * @category Mtc\Components\SagepayForm
 * @package  Mtc\Components\SagepayForm
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Components\SagepayForm;

use Config;
use Mtc\Shop\Abstracts\PaymentGateway;

/**
 * Sagepay Form Payment Gateway
 *
 * @category Mtc\Components\SagepayForm
 * @package  Mtc\Components\SagepayForm
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class SagepayForm extends PaymentGateway
{
    /**
     * Payment method title for the frontend.
     *
     * @var string
     */
    public $title = 'Sagepay';

    /**
     * The vendor password, defined within the config.
     *
     * @var string
     */
    protected $vendor_pass = '';

    /**
     * Retrieve the vendor password from the Config.
     *
     * @return void
     */
    public function __construct()
    {
        $this->vendor_pass = Config::get('sagepay-form.encryption_password');
        parent::__construct();
    }

    /**
     * Get a link to the transaction on the 3rd party gateway size (if applicable).
     *
     * @param Order $order The order object.
     *
     * @return string        Transaction URL, or empty string.
     */
    public function getTransactionUrl($order)
    {
        return route('sagepay-form.index');
    }

    /**
     * Encrypt and encode in a way Sagepay will be able to decrypt.
     *
     * @param string $strIn Unencrypted query string.
     *
     * @return string        Encrypted query string.
     */
    public function encryptAndEncode($strIn)
    {
        $strIn = $this->pkcs5_pad($strIn, 16);
        $pass = Config::get('sagepay-form.encryption_password');
        return "@".bin2hex(mcrypt_encrypt(MCRYPT_RIJNDAEL_128, $pass, $strIn, MCRYPT_MODE_CBC, $pass));
    }

    /**
     * Take Sagepay's encrypted query string and decode this into a format
     * we can read.
     *
     * @param string $strIn Encrypted query string.
     *
     * @return string        Decrypted query string.
     */
    private function decodeAndDecrypt($strIn)
    {
        $strIn = substr($strIn, 1);
        $strIn = pack('H*', $strIn);
        return mcrypt_decrypt(MCRYPT_RIJNDAEL_128, $this->vendor_pass, $strIn, MCRYPT_MODE_CBC, $this->vendor_pass);
    }

    /**
     * Add padding as required for mcrypt_decrypt
     *
     * @param string $text      Incoming text to pad.
     * @param int    $blocksize Number of blocks to pad by.
     *
     * @return string            Padded text.
     */
    private function pkcs5_pad($text, $blocksize)
    {
        $pad = $blocksize - (strlen($text) % $blocksize);
        return $text . str_repeat(chr($pad), $pad);
    }

    /**
     * Decode the incoming encrypted query string and return the result
     * as an unencrypted array of variables.
     *
     * @param string $strIn Encrypted query string.
     *
     * @return array         Decrypted array of variables.
     */
    public function decode($strIn)
    {
        $decodedString = $this->decodeAndDecrypt($strIn);
        // Convert the decoded query string into $sagepaayResponse.
        parse_str($decodedString, $sagepayResponse);
        return $sagepayResponse;
    }
}
