<?php
/**
 * Send Order Status Email Listener
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Listeners
 * @package  Mtc\Shop
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Listeners;

use Auth;
use Illuminate\Support\Facades\Mail;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Order\OrderNote;
use Mtc\Shop\Order\OrderStatus;
use Mtc\Shop\Events\OrderStatusUpdated;
use Mtc\Shop\Mail\OrderStatus as OrderStatusEmail;


/**
 * Providers a listener for the OrderStatusUpdated Event
 *
 * @category Mtc\Shop\Listeners
 * @package  Mtc\Shop
 * @author   James Neill <james.neill@mtcmedia.co.uk>
 */
class NotifyOrderStatusEmail
{
    /**
     * Handle the OrderStatusUpdated event
     *
     * @param OrderStatusUpdated $event
     * @return void
     * @internal param OrderContract $order
     *
     */
    public function handle(OrderStatusUpdated $event)
    {
        $order = $event->order;
        $sendable = OrderStatus::all()
            ->where('is_email_sent', 1)
            ->pluck('id')
            ->toArray();

        $this->addNote($order);

        if (in_array($order->status->id, $sendable)) {
            $this->sendEmail($order);
        }
    }

    /**
     * Appends a note to the order, detailing when
     * the status was changed and who by.
     *
     * @param $order
     */
    private function addNote(OrderContract $order)
    {
        $note = new OrderNote();
        $note->order_id = $order->id;
        $note->message = trans('shop::order.status_updated_note', [
            'new' => $order->status->title,
            'name' => Auth::user()->name,
            'id' => Auth::user()->id,
        ]);

        $note->save();
    }

    /**
     * Queues a new OrderStatus email
     *
     * @param $order
     */
    private function sendEmail(OrderContract $order)
    {
        Mail::to($order->email)
            ->queue(new OrderStatusEmail($order));
    }
}