<?php
/**
 * Per Product Save Listener
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods\Listeners
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods\Listeners;

use Mtc\Shop\Events\PriceUpdated;
use Mtc\Shop\PriceMethods\Models\PricePerProduct;
use Mtc\Shop\Product;
use Request;

/**
 * When a save has been initiated for a PerProduct price method, update
 * the price in the associated model.
 *
 * @category Mtc\Shop\PriceMethods\Listeners
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerProductSave
{
    /**
     * Load anything required - Laravel gets mad if this isn't here
     *
     * @return void
     */
    public function __construct()
    {

    }

    /**
     * Retrieve the data from the included form and update the price for the
     * product at it's various quantity levels.
     *
     * @param Product $product Product we are saving against.
     *
     * @return void
     */
    public function handle(Product $product)
    {
        $quantities = Request::input('price_per_product.quantity', []);
        $prices = Request::input('price_per_product.price', []);

        // Store a list of all saved price ids so we can delete any old ones.
        $saved_prices = [];

        collect(compact('quantities', 'prices'))
            ->transpose()
            // Ditch any whitespace
            ->map(
                function($item) {
                    return array_map('trim', $item);
                }
            )
            // Reject any empty inputs
            ->reject(
                function ($item) {
                    return empty($item[0]) || empty($item[1]);
                }
            )
            // Save the price and store in the $saved_prices aray
            ->each(
                function($item) use ($product, &$saved_prices) {
                    $price = PricePerProduct::firstOrNew(
                        [
                        'product_id' => $product->id,
                        'quantity' => $item[0],
                        ]
                    );

                    $price->price = intval($item[1] * 100);
                    $price->save();

                    $saved_prices[] = $price->id;
                }
            );

        // Delete any non-existing prices for this product
        PricePerProduct::whereProductId($product->id)
            ->whereNotIn('id', $saved_prices)
            ->delete();

        // Let the system know the price has been updated so sort price and
        // other things can be actioned.
        event(new PriceUpdated($product));
    }
}
