<?php
/**
 * Product Variant Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Nodeable;
use Mtc\Shop\Contracts\PriceMethod;

/**
 * Store information about a variant/size of a product.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Variant extends Model
{
    use Nodeable;

    /**
     * The attributes filled out by default when a new model is created.
     *
     * @var array
     */
    protected $attributes = [
        'stock_enabled' => false
    ];

    /**
     * The attributes that are mass assignable
     *
     * @var array
     */
    protected $fillable = [
        'product_id',
        'code',
        'stock_enabled',
        'stock_quantity',
        'shipping_weight',
        'shipping_dimensions_length',
        'shipping_dimensions_width',
        'shipping_dimensions_height'
    ];

    /**
     * Update parent model timestamps.
     *
     * @var array
     */
    protected $touches = ['product'];

    /**
     * The attributes that should be loaded when the model is loaded.
     *
     * @var array
     */
    protected $appends = ['price'];

    /**
     * Get the parent basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Get the base price for this variant
     *
     * @return int
     */
    public function getPriceAttribute()
    {
        return $this->prices->single();
    }

    /**
     * When price is requested via $variant->pricse, return the Price Method
     * for this product.
     *
     * @throws Exception If price method if not found
     *
     * @return \Mtc\Shop\Contracts\PriceMethod
     */
    public function getPricesAttribute() : PriceMethod
    {
        $price_method = $this->product->price_method;

        if (!class_exists($price_method)) {
            throw new \Exception("Price method not found: {$price_method}");
        }

        return new $price_method($this->product, $this);
    }
}
