<?php
/**
 * Basket Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use App\User;
use Illuminate\Database\Eloquent\Model;
use Mtc\Shop\Abstracts\PriceModifier;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Events\BasketTotal;
use Mtc\Shop\Variant;

/**
 * Base Basket model.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Basket extends Model implements BasketContract
{
    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'array',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'meta',
        'email',
        'phone',
    ];

    /**
     * Set the accessors to append to model arrays.
     *
     * @var array
     */
    protected $appends = [
        'cost_subtotal',
        'cost_tax',
        'cost_total',
    ];

    /**
     * Capture the boot event to save the basket_id within
     * the session if basket created.
     *
     * @return void
     */
    public static function boot()
    {
        parent::boot();

        static::created(
            function ($basket) {
                session(['basket_id' => $basket->id]);
            }
        );
    }

    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(BasketAddress::class);
    }

    /**
     * Get the list of items within the basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(BasketItem::class);
    }

    /**
     * Get the user this belongs to
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Add an item to the basket. If basket doesn't exist,
     * it will create it.
     *
     * @param Variant $variant  Variant model
     * @param integer $quantity Number of items to add to basket
     *
     * @return bool
     */
    public function addItem(Variant $variant, $quantity = 1)
    {
        // Get any existing item (if exists)
        $item = $this->items()
            ->where('variant_id', $variant->id)
            ->first() ?? new BasketItem;

        $item->variant_id = $variant->id;
        $item->quantity = $quantity;

        // If we don't have a basket yet, spool it up.
        if (!$this->exists) {
            $this->save();
        }

        return $this->items()->save($item);
    }

    /**
     * Get a value for a particular key in an address, first checking any
     * old data before showing data from the model.
     *
     * @param string $key  Key to check
     * @param string $type Address type to check (billing/shipping)
     *
     * @return string|null Value of key
     */
    public function getAddressValue($key, $type = 'billing')
    {
        if (old("addresses.{$type}.{$key}")) {
            return old("addresses.{$type}.{$key}");
        }

        $address = $this->addresses
            ->where('type', $type)
            ->first();

        if ($address) {
            return $address[$key];
        }

        return null;
    }

    /**
     * Get price modifiers for the final price
     *
     * @return \Illuminate\Support\Collection
     */
    public function getModifiersAttribute()
    {
        return collect(event(new BasketTotal($this)))
            ->filter(
                function ($item) {
                    return is_subclass_of($item, PriceModifier::class);
                }
            );
    }

    /**
     * Get the subtotal cost for all items in the basket, without tax.
     *
     * @return int
     */
    public function getCostSubtotalAttribute() : int
    {
        $total = 0;

        foreach ($this->items as $item) {
            $total += $item->variant->prices->multiple($item->quantity)->price * $item->quantity;
        }

        return $total;
    }

    /**
     * Get the total tax added to the items within the basket.
     *
     * @return int
     */
    public function getCostTaxAttribute() : int
    {
        $totals = [
            'price_excluding_tax' => 0,
            'price_including_tax' => 0,
        ];

        $items = [];

        foreach ($this->items as $item) {
            foreach (array_keys($totals) as $key) {
                $totals[$key] += $item->quantity * $item->variant
                    ->prices
                    ->multiple($item->quantity)
                    ->{$key};
            }
        }



        return abs(
            $totals['price_excluding_tax'] - $totals['price_including_tax']
        );
    }

    /**
     * Calculate the total, including any price modifiers.
     *
     * @return int
     */
    public function getCostTotalAttribute() : int
    {
        $total = $this->cost_subtotal;

        $this->modifiers->each(
            function ($item) use (&$total) {
                // @todo maybe modify this to work with Price class for tax
                // purposes?
                $total += $item->value;
            }
        );

        // Add tax to total if not inclusive
        if (false === \Config::get('tax.display_product_tax', true)) {
            $total += $this->cost_tax;
        }

        // Ensure the total is always > 0
        return $total > 0 ? $total : 0;
    }
}
