<?php
/**
 * Price Class
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Config;

/**
 * Price Class
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Price
{
    /**
     * The default price to show
     *
     * @var integer
     */
    public $price = 0;

    /**
     * The price without any discounts
     *
     * @var integer
     */
    public $price_original = 0;

    /**
     * The default price, without any tax if applicable
     *
     * @var integer
     */
    public $price_excluding_tax = 0;

    /**
     * The default price, including tax if applicable
     *
     * @var integer
     */
    public $price_including_tax = 0;

    /**
     * The config array set within config/tax
     *
     * @var array
     */
    protected $config = [];

    /**
     * The currently set tax rate, and it's applicable rates.
     *
     * @var array
     */
    protected $tax_rates = [];

    /**
     * Build a new Price instance, setting the defined attributes.
     *
     * @param int    $price    Price
     * @param string $tax_rate Key of rate from tax.rates
     */
    public function __construct(int $price, $tax_rate = 'Standard')
    {
        $this->config = Config::get('tax', []);

        if (array_key_exists($tax_rate, $this->config['rates'])) {
            $this->tax_rates = $this->config['rates'][$tax_rate];
        }

        $this->price_excluding_tax = $this->deductTax($price);
        $this->price_including_tax = $this->addTax($price);
        $this->price_original = $this->price_including_tax;

        if ($this->config['display_product_tax'] === false) {
            $this->price_original = $this->price_excluding_tax;
        }

        $this->price = $this->priceModifiers();
    }

    /**
     * Remove tax from the provided price if required.
     *
     * @param  int $price Price Value
     * @return int
     */
    protected function deductTax(int $price) : int
    {
        $tax_amount = 0;

        if (true === $this->config['enabled']
            && true === $this->config['price_entered_with_tax']
        ) {
            $tax_amount = $price - ( $price / ( $this->getTaxRate() + 1 ) );
        }
        return $price - $tax_amount;
    }

    /**
     * Add tax to the provided price if required.
     *
     * @param  int $price Price value
     * @return int
     */
    protected function addTax(int $price) : int
    {
        $tax_amount = 0;
        if (true === $this->config['enabled']
            && false === $this->config['price_entered_with_tax']
        ) {
            $tax_amount = $price * $this->getTaxRate();
        }
        return $price + $tax_amount;
    }

    /**
     * Deduct any discounts and the like from the orice.
     *
     * @return int
     */
    protected function priceModifiers() : int
    {
        // @todo make the event and the magic.
        return $this->price_original;
    }

    /**
     * Get the tax rate for the specific product.
     *
     * @return [type] [description]
     */
    protected function getTaxRate() : float
    {
        // @todo set this to work with multiple countries and matching against
        // the address entered.
        if (!empty($this->tax_rates)) {
            return (reset($this->tax_rates))['rate'];
        }

        return 0;
    }
}
