<?php
/**
 * Browse Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Mtc\Shop\Contracts\ProductRepositoryContract;
use Mtc\Shop\Product;
use Illuminate\Support\Collection;

/**
 * Actions revolving around displaying the browse page
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class BrowseController extends Controller
{
    /**
     * The options available by default for sorting items.
     *
     * @var array
     */
    public $sort_options = [
        [
            'key' => 'new-additions',
            'label' => 'New Additions'
        ],
        [
            'key' => 'price-asc',
            'label' => 'Price (Low-High)'
        ],
        [
            'key' => 'price-desc',
            'label' => 'Price (High-Low)'
        ],
    ];

    /**
     * Display the main page. Even though it's handled by Vue, we need to first
     * send the data syncronously otherwise Google and other bots are unable
     * to read the data.
     *
     * @param Request                   $request    Incoming request (used to
     *                                               check if JSON is wanted)
     * @param ProductRepositoryContract $repository Repository to run queries
     *                                               against
     * @param string                    $query      Search query string
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested
     *                                                         or regular view
     */
    public function index(Request $request, ProductRepositoryContract $repository, $query = '')
    {
        if (false == $request->wantsJson()) {
            $request->merge($this->convertQuery($query));
        }

        $json = $this->getBrowseData($request, $repository, $query);

        if (false === $request->wantsJson()) {
            $data = $this->convertQuery($query);
            $data['json'] = $json;

            // dd($data);
            return view('shop::public.browse')->with($data);
        } else {
            return $json;
        }
    }

    /**
     * Return the products, options, current URL, and sort options for this page
     *
     * @param Request                   $request    Incoming Request
     * @param ProductRepositoryContract $repository Repository to run queries
     * @param string                    $query      Search query string
     *
     * @return array Compacted list of items
     */
    protected function getBrowseData(
        Request $request,
        ProductRepositoryContract $repository,
        $query = ''
    ) {
        $products     = $repository->filter($request);
        $selected     = $this->getSelected($request->input('selected', []));
        $url          = $this->generateUrl($request);
        $sort_options = $this->sort_options;
        $options      = $this->getOptions($products, $selected);

        return compact('products', 'options', 'url', 'sort_options');
    }

    /**
     * Get the taxonomies to show on the filter
     *
     * @param  ProductRepository $products
     * @param  Collection        $selected
     * @return Collection
     */
    protected function getOptions($products, Collection $selected) : Collection
    {
        // Get the root taxonomies
        $roots = Taxonomy::roots()->with('children')
            ->get()
            ->map(function($root) use ($selected) {
                $root->selected = $selected->filter(function($item) use ($root) {
                    return $item->isSelfOrDescendantOf($root);
                });

                $last = $root->selected->last();

                if ($last !== null && $last->children->isEmpty() === false) {
                    $root = $last;
                } elseif ($last !== null) {
                    $root = $last->parent;
                }

                $root->children = $root->children->map(function($item) use ($selected) {
                    $item->selected = ! $selected->where('id', $item->id)->isEmpty();

                    $item->product_count = $item->getDescendantsAndSelf(['id'])->map(function($item) {
                        return $item->nodes()
                            ->whereNodeableType(Product::class)
                            ->public()
                            ->has('nodeable')
                            ->count();
                    })->sum();

                    return $item;
                });

                // Seems like ->reject() not working here?
                // Maybe because we're overriding the relationship?
                // This seems to fix annoyingly.
                $root = $root->toArray();
                $root['children'] = array_filter($root['children'], function($item) {
                    return $item['product_count'] > 0;
                });

                return $root;
            });

        return $roots;
    }

    /**
     * Take the incoming URL and parse this into the selected variables, sort,
     * and pagination.
     *
     * @param string $query Incoming query string
     *
     * @return array
     */
    protected function convertQuery($query) : array
    {
        // Set any defaults
        $data = [
            'query' => '',
            'selected' => collect([]),
            'sort_by' => '',
            'page' => 1,
            'full' => true,
        ];

        // Get the sort/page information
        preg_match_all("/\/?(sort|page)-([-_\w]+)?/", $query, $matches);
        collect($matches)->transpose()->each(
            function ($item) use (&$data) {
                $item[1] = ($item[1] == 'sort' ? 'sort_by' : $item[1]);
                $data[$item[1]] = $item[2];
            }
        );

        // + will be used to separate items within the same taxonomy
        $query = str_replace('+', '/', $query);
        // / will be used to seperate different terms
        $slugs = collect(explode('/', $query))
            ->reject(
                function ($item) {
                    // Remove sort/pagination (just in case)
                    return empty($item) || preg_match('/(?:sort|page)-.*/', $item) > 0;
                }
            );

        // If we have potential nodes, get information about them.
        if ($slugs->isEmpty() === false) {
            $data['selected'] = Node::select(['nodeable_id as id', 'title'])
                ->whereIn('slug', $slugs)
                ->get();
        }

        return $data;
    }

    /**
     * Generate the current URL or a URL containing additional options.
     *
     * @param Request $request  Incoming Request
     * @param array   $selected Optionally include an array to override
     *                           $request->input('selected')
     *
     * @return string Full URL (including site domain)
     */
    public function generateUrl(Request $request, $selected = [])
    {
        // Get a list of all IDs selected.
        $selected = empty($selected) ? $request->input('selected', []) : $selected;
        $ids = collect($selected)->pluck('id');
        $ids = Taxonomy::whereIn('id', $ids)->get();

        // Now find the root notes for each.
        $root_nodes = $ids->map(
            function ($child) {
                return $child->getRoot();
            }
        )->unique()->sortBy('id');

        // Determine sort of page options
        $additional = [];
        if ($request->input('sort_by')) {
            $additional[] = 'sort-' . $request->input('sort_by');
        }
        if ($request->input('page') && $request->input('page') > 1) {
            $additional[] = 'page-' . $request->input('page');
        }

        // Generate the URL
        return route('shop.browse') . '/' . $root_nodes->map(
            function ($root) use (&$ids) {
                return $ids->map(
                    function ($child, $child_index) use ($root, &$ids) {
                        if ($child->isDescendantOf($root)) {
                            $ids->forget($child_index);
                            return $child->node->slug;
                        }

                        return null;
                    }
                )->reject(
                    function ($child) {
                        return is_null($child);
                    }
                )->implode('+');
            }
        )->push($additional)->flatten()->implode('/');
    }

    /**
     * Get the selected taxonomies
     *
     * @param  mixed $selected
     * @return Collection
     */
    private function getSelected($selected) : Collection
    {
        // As this comes in an array with 'id' and 'title', only get the 'id'.
        $selected = (collect($selected))->pluck('id');

        if (!empty($selected)) {
            return Taxonomy::whereIn('id', $selected)->get();
        }

        return collect([]);
    }
}
