<?php
/**
 * Price Per Variant Price Method
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Mtc\Shop\Contracts\PriceMethod;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Create a 'Per Variant' price method.
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerVariant implements PriceMethod
{
    /**
     * Store the product model
     *
     * @var null|Product
     */
    protected $product = null;

    /**
     * Store the variant model
     *
     * @var null|Variant
     */
    protected $variant = null;

    /**
     * Setup the price method with the product and variant models.
     *
     * @param Mtc\Shop\Product      $product Product this is for
     * @param Mtc\Shop\Variant|null $variant Variant model
     *
     * @return void
     */
    public function __construct(Product $product, Variant $variant = null)
    {
        $this->product = $product;
        $this->variant = $variant;
    }

    /**
     * Get the current price for a single item
     *
     * @param integer $quantity (default: 1)
     *
     * @return integer
     */
    public function single()
    {
        return $this->multiple(1);
    }

    /**
     * Get the price for an item if the user requests the provided quantity
     *
     * @param integer $quantity (default: 1)
     *
     * @return integer
     */
    public function multiple(int $quantity = 1)
    {
        // Bulk pricing is built in by default.
        $price = PricePerVariant::where('variant_id', $this->variant->id)
            ->where('quantity', '<=', $quantity)
            ->orderBy('quantity', 'asc')
            ->first();

        return $price ? $price->price : null;
    }

    /**
     * Get the min/max price for an product.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float[] A 'min' and 'max' returned in an array.
     */
    public function range(int $quantity = 1)
    {
        $prices = PricePerVariant::where('variant_id', $this->variant->id)
            ->orderBy('price', 'asc')
            ->get();

        return [
            'min' => $prices->first(),
            'max' => $prices->last(),
        ];
    }
}
