<?php
/**
 * Basket Contract
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Contracts
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Contracts;

use Mtc\Shop\Variant;

/**
 * This interface defines the necessary methods for building a Basket
 *
 * @category Mtc\Shop\Contracts
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
interface BasketContract
{
    /**
     * Add an item to the basket. If basket doesn't exist, it will create it.
     *
     * @param Variant $variant  Variant Model
     * @param integer $quantity Number of items to add to basket
     *
     * @return bool
     */
    public function addItem(Variant $variant, $quantity);

    /**
     * Get the addresses associated with this basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses();

    /**
     * Get the subtotal cost for all items in the basket, without tax.
     *
     * @return float
     */
    public function getCostSubtotalAttribute();

    /**
     * Calculate the total, including any price modifiers.
     *
     * @return float
     */
    public function getCostTotalAttribute();

    /**
     * Get a value for a particular key in an address, first checking any
     * old data before showing data from the model.
     *
     * @param string $key  Key to check
     * @param string $type Address type to check (billing/shipping)
     *
     * @return string Value of key
     */
    public function getAddressValue($key, $type);

    /**
     * Get the list of items within the basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items();

    /**
     * Get the user this belongs to.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user();
}
