<?php
/**
 * Price Per Product Price Method
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Mtc\Shop\Contracts\PriceMethod;
use Mtc\Shop\PriceMethods\Price;
use Mtc\Shop\PriceMethods\Models\PricePerProduct;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Create a 'Per Product' price method.
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class PerProduct implements PriceMethod
{
    /**
     * Store the product model
     *
     * @var null|Product
     */
    protected $product = null;

    /**
     * Setup the price method with the product model.
     *
     * @param Mtc\Shop\Product      $product Stored against $this->product
     * @param Mtc\Shop\Variant|null $variant Ignored but required for Contract
     *
     * @return void
     */
    public function __construct(Product $product, Variant $variant = null)
    {
        $this->product = $product;
    }

    /**
     * Get the current price for a single item
     *
     * @param integer $quantity (default: 1)
     *
     * @return Price
     */
    public function single() : Price
    {
        return $this->multiple(1);
    }

    /**
     * Get the price for an item if the user requests the provided quantity
     *
     * @param integer $quantity (default: 1)
     *
     * @return Price
     */
    public function multiple(int $quantity = 1) : Price
    {
        $price = PricePerProduct::where('product_id', $this->product->id)
            ->where('quantity', '<=', $quantity)
            ->orderBy('quantity', 'asc')
            ->first();

        if ($price != null && $price->count() > 0) {
            return new Price($price->price ?: 0);
        }
        // what if no prices were defined?
        return new Price(0);
    }

    /**
     * Get the min/max price for an product.
     *
     * @param integer $quantity (default: 1)
     *
     * @return float[] A 'min' and 'max' returned in an array.
     */
    public function range(int $quantity = 1)
    {
        $prices = PricePerProduct::where('product_id', $this->product->id)
            ->orderBy('price', 'asc')
            ->get();

        return [
            'min' => new Price($prices->first()->price),
            'max' => new Price($prices->last()->price),
        ];
    }
}
