<?php
/**
 * Web Routes
 *
 * This file is where you may define all of the routes that are handled
 * by your application. Just tell Laravel the URIs it should respond
 * to using a Closure or controller method. Build something great!
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

use Mtc\Core\Admin\Menu;
use Mtc\Core\Admin\Tab;
use Mtc\Core\Events\Admin\MenuLoading;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Product;

$router->group(
    [
        'prefix' => 'admin',
        'middleware' => ['web', 'auth', 'permission:admin'],
        'as' => 'shop.admin.'
    ],
    function ($router) {
        $router->resource('products', Admin\ProductController::class);
        $router->resource('products/{product}/variants', Admin\VariantController::class);
        $router->resource('products/{product}/media', Admin\MediaController::class);

        $router->get('orders/export', 'Admin\OrderController@export')->name('orders.export');
        $router->post('orders/{order}/add_note', 'Admin\OrderController@addNote')->name('order.add_note');
        $router->resource('orders', 'Admin\OrderController');

        $router->group(
            [
                'prefix' => 'shipping',
                'as' => 'shipping.'
            ],
            function ($router) {
                $router->resource('zones', 'Admin\ShippingZoneController');
                $router->resource('flat-rate', 'Admin\ShippingFlatRateController');
            }
        );
    }
);

$router->group(
    ['middleware' => ['web']], function ($router) {

        $router->match(
            ['get', 'post'],
            trans('shop::browse.prefix') . '/{query?}',
            BrowseController::class . '@index'
        )->name('shop.browse')
            ->where('query', '.*');

        $router->get('product/{product}/{slug?}', ProductController::class . '@show')->name('shop.product');

        $router->group(
            [
            // By passing through the trans() we can easily customise this route through lang files.
            'prefix' => trans('shop::basket.prefix'),
            'as' => 'shop.basket.'
            ], function ($router) {

                $router->post('/', 'BasketController@store')->name('store');


                $router->get('/', 'BasketController@index')->name('index');
                $router->get('overview', 'BasketController@overview')->name('overview');
                $router->post('overview/confirm', 'BasketController@confirm')->name('overview.confirm');
                $router->put('/update', 'BasketController@update')->name('update');
            }
        );

    }
);

Event::listen(
    MenuLoading::class, function () {
        Menu::addSection('Shop', 'shop');
        Menu::addMenu('shop', 'Products', '', 'products', 'glyphicon glyphicon-barcode');
        Menu::addSubMenu('products', 'Manage Products', '', route('shop.admin.products.index'), 'manage_products');

        Menu::addMenu(
            'shop',
            'Orders',
            '',
            'orders',
            'glyphicon glyphicon-briefcase'
        );
        Menu::addSubMenu(
            'orders',
            'Manage Orders',
            '',
            route('shop.admin.orders.index'),
            'manage_orders'
        );

        Menu::addMenu(
            'shop',
            'Shipping',
            '',
            'shipping',
            'glyphicon glyphicon-plane'
        );
        Menu::addSubMenu(
            'shipping',
            'Zones',
            '',
            route('shop.admin.shipping.zones.index'),
            'manage_shipping_zones'
        );
        Menu::addSubMenu(
            'shipping',
            'Flat Rates',
            '',
            route('shop.admin.shipping.flat-rate.index'),
            'manage_shipping_rates'
        );
    }
);

Event::listen(
    'tabs.loading: ' . Product::class, function ($model) {
        // if no model id only show tabs that relate to product without ID
        if (empty($model->id)) {
            return [
                new Tab(
                    'products.create',
                    'Details',
                    route('shop.admin.products.create', $model)
                )
            ];
        }
        return [
            new Tab(
                'products.edit',
                'Details',
                route('shop.admin.products.edit', $model)
            ),
            new Tab(
                'variants.index',
                'Variants',
                route('shop.admin.variants.index', $model)
            ),
            new Tab(
                'media.index',
                'Media',
                route('shop.admin.media.index', $model)
            )
        ];
    }
);

Event::listen(
    'shop.shipping_methods', function() {
        return new Mtc\Shop\ShippingMethods\FlatRate;
    }
);
