<?php

namespace Mtc\Shop\Http\Controllers\Admin;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Shop\ShippingZone;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;

/**
 * Class ShippingZoneController
 *
 * @package Mtc\Shop\Http\Controllers\Admin
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ShippingZoneController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = ShippingZone::orderBy('id', 'DESC');

        return (new Builder('shop.admin.shipping.zones', $query))->view();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param ShippingZone $zone
     * @return \Illuminate\View\View
     */
    public function create(ShippingZone $zone)
    {
        // Get a list of countries and states.
        $countries = \Mtc\Core\Country::whereStatus(1)
            ->with('states')
            ->get();

        // Throw the contries in here so we can acccess it through the builder.
        $zone->_countries = $countries;

        return (new ItemBuilder('shop.admin.shipping.zones', $zone))
            ->view('shop::admin.zones.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request
     * @param ShippingZone $zone
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request, ShippingZone $zone)
    {
        $this->validate($request, [
            'title' => 'required|unique:shipping_zones|max:255',
            'regions' => 'array'
        ]);

        // Take an array of values like country_1 and transform into a
        // collection with type and id.
        $regions = collect($request->input('regions', []))
            ->map(function ($item) {
                $item = explode('_', $item);
                return [
                    'type' => $item[0],
                    'id' => (int)$item[1]
                ];
            });

        $zone->fill($request->all())
            ->save();

        // Loop through the regions and relations to store the result
        foreach (['countries', 'states'] as $relation) {
            $zone->$relation()->sync(
                $regions->where('type', str_singular($relation))
                    ->pluck('id')
                    ->toArray()
            );
        }

        return redirect(route('shop.admin.shipping.zones.index'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param ShippingZone $zone
     * @return \Illuminate\View\View
     */
    public function edit(ShippingZone $zone)
    {
        return $this->create($zone);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param ShippingZone $zone
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(Request $request, ShippingZone $zone)
    {
        return $this->store($request, $zone);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request
     * @param ShippingZone $zone
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function destroy(Request $request, ShippingZone $zone)
    {
        $zone->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "Zone '{$zone->title}' has been deleted.");
        return redirect()->back();
    }
}
