<?php

namespace Mtc\Shop\ShippingMethods;

use Illuminate\Support\Collection;
use Mtc\Shop\Abstracts\ShippingMethod;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Http\Controllers\Admin\ShippingFlatRateController;
use Mtc\Shop\ShippingFlatRate;

/**
 * Class FlatRate
 *
 *
 * @package Mtc\Shop\ShippingMethods
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 * @version VERSION
 */
class FlatRate extends ShippingMethod
{
    /**
     * @var string $id Shipping Method identificator
     */
    public $id = 'flat-rate';

    /**
     * @var string $title Shipping Method title
     */
    public $title = 'Flat Rate';

    /**
     * Calculagte shipping costs
     *
     * WIP
     *
     * @param BasketContract|null $basket
     * @return array
     */
    public function calculateShipping(BasketContract $basket = null): array
    {
        /*
         * What do I need to know?
         *
         * 1. The shipping/billing zone.
         * 2. The total of the basket.
         * 3. The delivery methods.
         * 3. Return in a friendly interface.
         */

        $total = 100;
        $zone = 1;

        return ShippingFlatRate::whereShippingZoneId(1)
            ->where('min_basket_value', '<=', $total)
            ->where(function ($query) use ($total) {
                $query->whereNull('max_basket_value')
                    ->orWhere('max_basket_value', '>=', $total);
            })
            ->get()
            ->sortBy('value')
            ->keyBy('id')
            ->toArray();
    }

    /**
     * Flat rates always can be edited/added
     *
     * @return string
     */
    public function isManageable()
    {
        return true;
    }

    /**
     * Get manage url for a flat rate method management (list / edit view)
     *
     * @param int $zone_id Selected Zone whose rates to display
     * @param int $rate_id Specific rate to edit
     * @return string url to action
     */
    public function manageUrl($zone_id, $rate_id = false): string
    {
        // Find Rate
        $rate = ShippingFlatRate::find($rate_id);

        // If we have rate we can open single rate view
        if ($rate) {
            return route(
                ShippingFlatRateController::$route_name . ".edit",
                [
                    'zone' => $zone_id,
                    'rate' => $rate->id
                ]
            );
        }

        // return index page
        return route(
            ShippingFlatRateController::$route_name . ".index",
            [
                'zone' => $zone_id
            ]
        );
    }

    /**
     * Get a breakdown for Flat rates for a specific zone
     *
     * @param int $shipping_zone_id
     * @return mixed
     */
    public function getCostBreakdown(int $shipping_zone_id): Collection
    {
        return ShippingFlatRate::where('shipping_zone_id', $shipping_zone_id)
            ->orderBy('min_basket_value', 'asc')
            ->get();
    }
}
