<?php
/**
 * Basket Tests
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Tests
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithoutMiddleware;
use Mtc\Core\Node;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Run tests to ensure proper execution of the basket.
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Tests
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class BasketTest extends TestCase
{
    use DatabaseTransactions;

    /**
     * Check that the basket is resolved.
     *
     * @return void
     */
    public function testBasketExists()
    {
        $basket = resolve(BasketContract::class);
        $this->assertTrue($basket instanceof BasketContract);
    }

    /**
     * Ensure the basket hook is triggered to set the
     * basket_id for the current session.
     *
     * @return void
     */
    public function testBasketWillSaveSessionId()
    {
        $this->assertNull(session('basket_id'));

        $basket = resolve(BasketContract::class);
        $basket->save();

        $this->assertEquals(session('basket_id'), $basket->id);
    }

    /**
     * Ensure an empty basket returns an empty result.
     *
     * @return void
     */
    public function testEmptyBasket()
    {
        $basket = resolve(BasketContract::class);
        $this->assertEquals($basket->items->count(), 0);

        $message = trans('shop::basket.empty');
        $this->visit(route('shop.basket.index'))
            ->see($message);
    }

    /**
     * Confirm that the basket can add variants and adding
     * the same variant will simply increase quantities.
     *
     * @return void
     */
    public function testBasicVariantAdd()
    {
        $basket = resolve(BasketContract::class);
        // Turning this off temp.
        return;

        // Create a product
        $product = new Product;
        $product->save();
        $product->node()->save(
            new Node(
                [
                    'title' => 'Title',
                    'description' => 'Description',
                ]
            )
        );

        // Create a variant
        $variant = new Variant;
        $product->variants()->save($variant);
        $variant->node()->save(
            new Node(
                [
                    'title' => 'Title',
                    'description' => 'Description',
                ]
            )
        );

        // Add to the item
        $basket->addItem($variant);

        $this->assertEquals($basket->items()->count(), 1);

        $basket->addItem($variant);
        $this->assertEquals($basket->items()->count(), 1);
        $this->assertEquals($basket->items()->sum('quantity'), 2);
    }
}