<?php

use Faker\Factory;
use Illuminate\Database\Seeder;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

class ProductSeeder extends Seeder
{

    protected $factory = null;
    protected $taxonomies = [];

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $this->factory = Factory::create();
        $this->taxonomies = Taxonomy::where('parent_id', '>', 0)
            ->pluck('id')
            ->toArray();

        // Get a definitive amount of products to add.
        $amount = false;
        while ($amount === false) {
            $amount = $this->command->ask("How many products?", 500);
            $amount = filter_var($amount, FILTER_VALIDATE_INT);
        }

        $bar = $this->command->getOutput()->createProgressBar($amount);

        for ($i = 0; $i < $amount; $i++) {
            $this->addProduct();
            $bar->advance();
        }

        $bar->finish();
    }

    protected function addProduct()
    {
        $product = new Product;
        $node = $this->newNode();

        // Set the request to include the price per product
        Request::merge($this->getPricePerProduct());
        $product->save();

        $product->node()->save($node);

        // Add random taxonomies
        $tax = (collect(array_rand($this->taxonomies)))
            ->take(rand(1, 3))
            ->reject(
                function ($value) {
                    return $value == 0;
                }
            )
            ->toArray();

        $node->taxonomies()->attach($tax);

        for ($v = 0; $v < rand(1, 5); $v++) {
            $this->addVariant($product);
        }
    }

    protected function addVariant(Product $product)
    {
        $variant = new Variant(
            [
            'product_id' => $product->id,
            'code' => $this->factory->ean13,
            ]
        );
        $variant->save();
        $variant->node()->save($this->newNode());
    }

    protected function newNode()
    {
        return new Node(
            [
            'title' => $this->factory->sentence(5),
            'description' => "<p>" . implode("</p><p>", $this->factory->paragraphs(4)) . "</p>",
            'status' => 'published',
            'visibility' => 'public',
            ]
        );
    }

    protected function getPricePerProduct()
    {
        return [
            'price_per_product' => [
                'quantity' => [1],
                'price' => [$this->factory->randomFloat(2, 0.01, 500)],
                'tax' => 'Standard',
            ],
        ];
    }
}
