<?php
/**
 * Basket Address Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Illuminate\Database\Eloquent\Model;
use Mtc\Shop\Contracts\BasketContract;

/**
 * Model storing information about an basket's address.
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class BasketAddress extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'basket_id',
        'type',
        'first_name',
        'last_name',
        'address1',
        'address2',
        'city',
        'state',
        'postcode',
        'country',
    ];

    /**
     * Update parent model timestamps.
     *
     * @var array
     */
    protected $touches = ['basket'];

    /**
     * Get the parent basket.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basket()
    {
        return $this->belongsTo(Basket::class);
    }

    /**
     * Validation rules for an address
     *
     * @param string $type Address type
     * @param string $prefix
     * @return array
     */
    public static function getAddressValidationRules($type = 'billing', $prefix = 'addresses'): array
    {
        // TODO: implement country specific checks if postcode is required
        // TODO: regex for country specific postcode
        $postcode_required = true;
        return [
            "$prefix.$type.first_name" => 'required',
            "$prefix.$type.last_name" => 'required',
            "$prefix.$type.address1" => 'required',
            "$prefix.$type.city" => 'required',
            "$prefix.$type.postcode" => 'alpha_num' . ($postcode_required ? '|required' : ''),
            "$prefix.$type.country" => 'required',
        ];
    }

    /**
     * Validation rules field names for an address
     *
     * @param string $type Address type
     * @param string $prefix
     * @return array
     */
    public static function getAddressValidationFieldNames($type = 'billing', $prefix = 'addresses'): array
    {
        return [
            "$prefix.$type.first_name" => trans('First Name'),
            "$prefix.$type.last_name" => trans('Last Name'),
            "$prefix.$type.address1" => trans('Address'),
            "$prefix.$type.city" => trans('City'),
            "$prefix.$type.postcode" => trans('Postcode'),
            "$prefix.$type.country" => trans('Country')
        ];
    }
}
