<?php
/**
 * Variant Admin Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Http\Requests;
use Mtc\Core\Node;
use Mtc\Shop\Http\Requests\Admin\StoreVariant;
use Mtc\Shop\Product;
use Mtc\Shop\Variant;

/**
 * Displays actions for Variants as a resource.
 *
 * @category Mtc\Shop\Http\Controllers\Admin
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class VariantController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Product $product Parent model
     *
     * @return \Illuminate\View\View
     */
    public function index(Product $product)
    {
        return (new ItemBuilder('core.admin.product', $product, $product->node))
            ->tab(
                'shop::admin.variants.index', [
                    'variants' => Variant::whereProductId($product->id)->paginate(),
                    'buttons' => [
                        [
                            'label' => 'Add Variant',
                            'href'  => route('shop.admin.variants.create', $product),
                            'class' => 'btn btn-primary',
                        ]
                    ]
                ]
            );
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param Product $product Parent model
     *
     * @return \Illuminate\View\View
     */
    public function create(Product $product)
    {
        return (
            new ItemBuilder(
                'shop.admin.variants',
                new Variant(['product_id' => $product->id]),
                new Node
            ))
            ->routeParams([$product->id])
            ->view('shop::admin.variants.form');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreVariant $request Validated request
     * @param Product      $product Parent model
     * @param Variant      $variant Empty Variant model to be populated
     * @param Node         $node    Empty Node model to be populated
     *
     * @return \Illuminate\Http\Response
     * @throws \Illuminate\Database\Eloquent\MassAssignmentException
     */
    public function store(StoreVariant $request, Product $product, Variant $variant, Node $node)
    {
        $variant->product_id = $product->id;
        $variant->fill($request->input('variant', []))->save();
        $node->fill($request->input('node', []));
        $variant->node()->save($node);

        return redirect(route('shop.admin.variants.index', $product));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Product $product Parent model
     * @param Variant $variant Variant model
     *
     * @return \Illuminate\View\View
     */
    public function edit(Product $product, Variant $variant)
    {
        return (new ItemBuilder('shop.admin.variants', $variant, $variant->node))
            ->routeParams([$product->id])
            ->view('shop::admin.variants.form');
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreVariant $request Validated request
     * @param Product      $product Parent model
     * @param Variant      $variant Variant model
     *
     * @return \Illuminate\Http\Response
     * @throws \Illuminate\Database\Eloquent\MassAssignmentException
     */
    public function update(StoreVariant $request, Product $product, Variant $variant)
    {
        $variant->fill($request->input('variant', []))->save();
        $variant->node->fill($request->input('node', []));
        $variant->node->save();

        return redirect(route('shop.admin.variants.index', $product));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request Incoming Request
     * @param Product $product Parent model
     * @param Variant $variant Variant model to be removed
     *
     * @return \Illuminate\Http\Response Either JSON, or a redirect back.
     * @throws \Exception
     */
    public function destroy(Request $request, Product $product, Variant $variant)
    {
        if ($variant->node) {
            $variant->node->delete();
        }

        $variant->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "{$variant->title} has been deleted.");
        return redirect()->back();
    }
}
