<?php
/**
 * Basket Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Illuminate\Http\Request;
use Mtc\Core\Country;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Events\RetrievePaymentGateways;
use Mtc\Shop\Http\Requests\StoreBasketInfo;
use Mtc\Shop\Http\Requests\StoreBasketItems;
use Mtc\Shop\Variant;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Actions revolving around Basket creation and update.
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class BasketController extends Controller
{
    /**
     * Display the basket.
     *
     * @param Request        $request Incoming request used to check if wanting JSON.
     * @param BasketContract $basket  Basket model
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise View.
     */
    public function index(Request $request, BasketContract $basket)
    {
        $basket->items = $this->getProcessedItems($basket);

        $basket->action = route('shop.basket.store');
        $basket->editable = true;

        if ($request->wantsJson()) {
            // if there's a warning on the session but we're outputting JSON for an ajax request
            // put the warning into our response...
            if ($request->session()->has('warning')) {
                $basket->warning = $request->session()->get('warning');
            }
            return $basket;
        }

        // Get a list of countries, with the default on top.
        $countries = Country::whereStatus(1)->get()
            ->sortBy(
                function ($item, $index) {
                    if ($item->code == env('DEFAULT_COUNTRY', 'GB')) {
                        $index = -1;
                    }

                    return $index;
                }
            );

        return view('shop::public.basket')
            ->with(compact('basket', 'countries'));
    }

    /**
     * Save items to a basket.
     *
     * @param StoreBasketItems $request Validated Request
     * @param BasketContract   $basket  Basket Model
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested
     *                                                         or just a view.
     */
    public function store(StoreBasketItems $request, BasketContract $basket)
    {
        foreach ($request->input('items') as $item) {
            $variant = Variant::find($item['variant_id']);
            if ($variant->stock_quantity >= $item['quantity']) {
                $basket->addItem($variant, $item['quantity']);
            } else {
                // There aren't enough of them! Add what we can to the basket
                $basket->addItem($variant, $variant->stock_quantity);
                $warning = "We don't have quite enough of these in stock right now. "
                            . $variant->stock_quantity
                            . " units have been added to your basket.";
            }
        }

        $basket->items()->where('quantity', '<', 1)->delete();

        $request->session()->reflash();
        if (isset($warning)) {
            $request->session()->flash('warning', $warning);
        }
        $request->session()->flash('previous', $variant->product->getUrl());

        return $this->index($request, $basket);
    }

    /**
     * Return a list of items associated with this basket with their URL,
     * price per unit and total price per line (excluding tax)
     *
     * @param BasketContract $basket Basket model
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getProcessedItems(BasketContract $basket)
    {
        return $basket->items()
            ->with(
                [
                'variant',
                'variant.node',
                'variant.product',
                'variant.product.node',
                ]
            )->get()->map(
                function ($line) {
                        $line->url = $line->variant->product->getUrl();
                        $line->price_unit = $line->variant
                            ->prices
                            ->multiple($line->quantity)
                            ->price;
                        $line->price_total = $line->price_unit * $line->quantity;
                        return $line;
                }
            );
    }
}
