<?php
/**
 * Basket Controller
 *
 * PHP Version 7
 *
 * @package  Mtc\Shop
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Event;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Country;
use Mtc\Shop\Abstracts\ShippingMethod;
use Mtc\Shop\BasketAddress;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Events\RetrievePaymentGateways;

/**
 * Class CheckoutController
 *
 * Checkout process controller.
 * Manages process of checkout with 3 steps
 * - customer info (email & phone) and shipping details
 * - shipping method
 * - payment method & billing details
 *
 * @package  Mtc\Shop
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class CheckoutController extends Controller
{
    /**
     * Display the checkout first step - customer details.
     *
     * @param Request $request Incoming request used to check if wanting JSON.
     * @param BasketContract $basket Basket model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise View.
     */
    public function customer(Request $request, BasketContract $basket)
    {
        $basket->items = $this->getProcessedItems($basket);

        // Check if user has items in basket, redirect to basket if empty
        if (count($basket->items) === 0) {
            return redirect(action('\\' . BasketController::class . '@index'));
        }

        $basket->editable = true;
        $basket->action = action('\\' . self::class . '@customerStore');
        $basket->step = 'customer';

        if ($request->wantsJson()) {
            return $basket;
        }

        // Get a list of countries, with the default on top.
        $countries = Country::getOrderedActiveCountries();

        return view('shop::public.checkout')
            ->with(compact('basket', 'countries'));
    }

    /**
     * Save customer default info + shipping details to a basket.
     *
     * @param Request $request Incoming Request
     * @param BasketContract $basket Basket Model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested or just a view.
     */
    public function customerStore(Request $request, BasketContract $basket)
    {
        // Main validation rules
        $validation_rules = [
            'email' => 'required|email',
            'phone' => 'numeric'
        ];

        // Validate incoming request
        $this->validate($request, array_merge(
            $validation_rules,
            BasketAddress::getAddressValidationRules('shipping')
        ), [], BasketAddress::getAddressValidationFieldNames('shipping'));

        // Fill in info about for basket
        $basket->fill($request->all());
        $basket->save();

        // Save shipping address
        if ($basket->shippingAddress) {
            $basket->shippingAddress
                ->update($request->input('addresses.shipping'));
        } else {
            $address = $basket->shippingAddress()
                ->create($request->input('addresses.shipping'));

            // We need to set address type
            $address->type = 'shipping';
            $address->save();
        }

        return redirect(action('\\' . self::class . '@shipping'));
    }

    /**
     * Display the shipping method selection.
     *
     * @param Request $request Incoming request used to check if wanting JSON.
     * @param BasketContract $basket Basket model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise View.
     */
    public function shipping(Request $request, BasketContract $basket)
    {
        // redirect back if user does not have a shipping address set to the basket
        if (!$basket->shippingAddress) {
            return redirect(action('\\' . self::class . '@customer'));
        }

        $basket->items = $this->getProcessedItems($basket);
        $basket->editable = true;
        $basket->action = action('\\' . self::class . '@shippingStore');
        $basket->step = 'shipping';
        $basket->shipping_methods = $this->getAvailableShippingMethods($basket);

        if ($request->wantsJson()) {
            return $basket;
        }

        // Get a list of countries, with the default on top.
        $countries = Country::getOrderedActiveCountries();

        return view('shop::public.checkout')
            ->with(compact('basket', 'countries'));
    }

    /**
     * Save shipping method to a basket.
     *
     * @param Request $request Incoming Request
     * @param BasketContract $basket Basket Model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested or just a view.
     */
    public function shippingStore(Request $request, BasketContract $basket)
    {
        // Validation required on POST request only
        if (!$request->wantsJson()) {
            // Make sure shipping address is validated
            $this->validate($request, [
                'shipping_method' => 'required'
            ]);
        }

        // Fetch all available shipping methods
        $shipping_methods = $this->getAvailableShippingMethods($basket);

        // Find the method that was chosen
        $chosen_method = $shipping_methods
            ->where('id', $request->input('basket.shipping_method_id'))
            ->first();

        // Update basket to have the chosen shipping method
        if ($chosen_method) {
            $basket->cost_shipping = $chosen_method['value'];
            $basket->shipping_method_id = $chosen_method['id'];
            $basket->save();
        }

        // Update basket to have all required information for the basket display
        $basket->shipping_methods = $shipping_methods;
        $basket->action = action('\\' . self::class . '@shippingStore');
        $basket->step = 'shipping';
        $items = $this->getProcessedItems($basket);
        $basket->items = $items;

        // if this was an ajax triggered by input change, return basket
        if ($request->wantsJson()) {
            // We will need to process items for making sure prices are displayed correctly when ajax reloads
            return $basket;
        }

        return redirect(action('\\' . self::class . '@payment'));
    }

    /**
     * Display the basket payment options and allow user to change billing details.
     *
     * @param Request $request Incoming request used to check if wanting JSON.
     * @param BasketContract $basket Basket model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise View.
     */
    public function payment(Request $request, BasketContract $basket)
    {
        // Redirect back to shipping method selection when basket has no shipping method attached
        if (empty($basket->shipping_method_id)) {
            return redirect(action('\\' . self::class . '@shipping'));
        }

        $basket->items = $this->getProcessedItems($basket);
        $basket->editable = true;
        $basket->action = action('\\' . self::class . '@paymentStore');
        $basket->step = 'payment';

        // we need to display the shipping method name on saved details
        $selected_shipping_method = $this->getAvailableShippingMethods($basket)
            ->where('id', $basket->shipping_method_id)
            ->first();

        $basket->shipping_method = $selected_shipping_method['title'] ?? '';

        // Get a list of registered payment gateways.
        $basket->payment_methods = $this->getAvailablePaymentMethods($basket);

        if ($request->wantsJson()) {
            return $basket;
        }

        // Get a list of countries, with the default on top.
        $countries = Country::getOrderedActiveCountries();

        return view('shop::public.checkout')
            ->with(compact('basket', 'countries'));
    }

    /**
     * Save items to a basket.
     *
     * @param Request $request Incoming Request
     * @param BasketContract $basket Basket Model
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested or just a view.
     */
    public function paymentStore(Request $request, BasketContract $basket)
    {
        if ($request->wantsJson()) {
            if ($request->input('action') === 'same_billing_address') {
                // If Same address checkbox was triggered
                $basket->same_billing_address = $request->input('basket.same_billing_address');
                $basket->save();
            } elseif ($request->input('action') === 'set_payment_method') {
                // If payment method change was triggered
                $basket->payment_method = $request->input('basket.payment_method');
                $basket->save();
            }

            // Get a list of registered payment gateways and make sure items have their prices loaded
            $basket->payment_methods = $this->getAvailablePaymentMethods($basket);
            $basket->items = $this->getProcessedItems($basket);

            // Return ajax response
            return $basket;
        }

        // Set up validation
        $validation_rules = [
            'payment_method' => 'required',
            'terms' => 'required',
        ];

        $validation_field_names = [];

        // If user has selected to use a different billing address we need to add new rules
        if (empty($request->input('same_billing_address'))) {
            // Add address rules
            $validation_rules = array_merge(
                $validation_rules,
                BasketAddress::getAddressValidationRules('billing')
            );

            $validation_field_names = array_merge(
                $validation_field_names,
                BasketAddress::getAddressValidationFieldNames('billing')
            );
        }

        // Get a list of registered payment gateways.
        $basket->payment_methods = $this->getAvailablePaymentMethods($basket);

        // execute validation
        $this->validate($request, $validation_rules, [], $validation_field_names);

        // If we are using the same address as shipping address just clone the necessary data
        if (!empty($request->input('same_billing_address')) && count($basket->billingAddress) === 0) {
            $address = $basket->shippingAddress->toArray();
            $address['type'] = 'billing';
            unset($address['id']);
            $basket->billingAddress()
                ->create($address);
        } else {
            // load address data to a variable
            $address = $request->input('addresses.billing');

            // Check if billing address exists
            if (!empty($basket->billingAddress)) {
                // Update existing record
                $basket->billingAddress()
                    ->update($address);
            } else {
                // Create new record for billing address
                // We need to pre-fill address type
                $address['type'] = 'billing';
                $basket->billingAddress()
                    ->create($address);
            }
        }

        // Reload relationship value for basket object
        $basket->load('billingAddress');

        // if this was an ajax triggered by input change, return basket
        if ($request->wantsJson()) {
            return $basket;
        }

        $basket->shipping_methods = $this->getAvailableShippingMethods($basket);

        // Generate a new order from basket. redirect back to basket if failed
        $order = resolve(OrderContract::class)->createFromBasket($basket);
        if (!$order) {
            $request->session()->flash(
                'message',
                trans('Failed to create order, please make sure your basket is filled properly')
            );
            return redirect(action('\\' . self::class . '@payment'));
        }

        // Save order ID in session
        $request->session()->put('order_id', $order->id);

        // Get the chosen payment gateway
        $class = $request->input('payment_method');

        // Make sure we can trigger payment gateway
        if (!class_exists($class) && method_exists(new $class, 'getTransactionUrl')) {
            if (!$order) {
                $request->session()->flash(
                    'message',
                    trans('Please make sure you have chosen a payment gateway')
                );
                return redirect(action('\\' . self::class . '@payment'));
            }
        }
        // return the transaction url
        return redirect($class::getTransactionUrl($order));
    }

    /**
     * Return a list of items associated with this basket with their URL,
     * price per unit and total price per line (excluding tax)
     *
     * @param BasketContract $basket Basket model
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getProcessedItems(BasketContract $basket)
    {
        return $basket->items()
            ->with([
                'variant',
                'variant.node',
                'variant.product',
                'variant.product.node',
            ])->get()->map(
                function ($line) {
                    $line->url = $line->variant->product->getUrl();
                    $line->price_unit = $line->variant
                        ->prices
                        ->multiple($line->quantity)
                        ->price;
                    $line->price_total = $line->price_unit * $line->quantity;
                    return $line;
                }
            );
    }

    /**
     * Get Available Shipping Methods based on given basket
     * Call all Shipping methods registered, get rates from them and return the rates on a single level collection
     *
     * @param BasketContract $basket current user basket
     * @return Collection single level collection of all available rates
     */
    protected function getAvailableShippingMethods(BasketContract $basket): Collection
    {
        /*
         * Fetch all registered shipping methods into a collection
         * Each Shipping Method must have a method named calculateShipping which returns viable rates
         * At the end lets flatten by one level to ensure all rates are on the same level in one array.
         */
        return collect(Event::fire('shop.shipping_methods'))
            ->map(function (ShippingMethod $method) use ($basket) {
                return collect($method->calculateShipping($basket))
                    ->map(function ($rate) use ($method) {
                        /*
                         * We need to make sure that Delivery methods have unique IDs across all types
                         * To make this happen we simply update Rate ID by pre-pending Method ID
                         */
                        $rate['method_specific_id'] = $rate['id'];
                        $rate['id'] = $method->id . '-' . $rate['id'];
                        return $rate;
                    });
            })->flatten(1);
    }

    /**
     * Retrieve list of available payment methods to the basket
     * This retrieves all payment gateway instances and sets the ID as payment gateway class name
     *
     * @param BasketContract $basket users basket
     * @return Collection list of payment gateways available for this basket
     */
    protected function getAvailablePaymentMethods(BasketContract $basket): Collection
    {
        // Get a list of registered payment gateways.
        return collect(event(new RetrievePaymentGateways($basket)))
            ->map(function ($method) {
                // make sure we have an identifier for the method
                $method->id = get_class($method);
                return $method;
            });
    }
}
