<?php
/**
 * Class ShippingFlatRateController
 *
 * @package Mtc\Shop\Http\Controllers\Admin
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers\Admin;

use Illuminate\Support\Facades\Event;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Admin\Builder;
use Mtc\Shop\ShippingZone;
use Mtc\Shop\ShippingFlatRate as Model;

/**
 * Class ShippingFlatRateController
 *
 * Controller to manage Flat Rates.
 *
 * @package Mtc\Shop\Http\Controllers\Admin
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class ShippingFlatRateController extends Controller
{

    /**
     * @var string $route_name path to this controller
     */
    public static $route_name = 'shop.admin.shipping.flat-rate';

    /**
     * Display a listing of the resource.
     *
     * @param ShippingZone $zone Zone under which this method lies
     * @return \Illuminate\Http\Response
     */
    public function index(ShippingZone $zone, Request $request)
    {
        if (!$zone->id && $request->input('zone')) {
            $zone = ShippingZone::find($request->input('zone'));
        }

        Event::listen('core.builder.shop.admin.shipping.flat-rate.head', function () {
            return 'shop::admin.shipping.flat_rate.filter';
        });
        // Set up query
        $query = Model::query();

        // if we have filtered by zone, only display methods of this zone
        if ($zone->id) {
            $query = $query->where('shipping_zone_id', $zone->id);
        }

        // Order by basket value
        $query = $query->orderBy('min_basket_value', 'asc');

        $builder = (new Builder(self::$route_name, $query));

        // if we have zone set, add the zone id to url params
        if ($zone->id) {
            $builder = $builder->routeParams([$zone->id]);
        }

        return $builder->columns([
                'title' => trans('fields.title'),
                'value' => trans('Value'),
                'min_basket_value' => trans('Min Basket Value'),
                'max_basket_value' => trans('Max Basket Value')
            ])
            ->data([
                'value' => function ($rate) {
                    return $rate->value / 100;
                },
                'min_basket_value' => function ($rate) {
                    return $rate->min_basket_value / 100;
                },
                'max_basket_value' => function ($rate) {
                    return $rate->max_basket_value / 100;
                },
            ])
            ->view([
                'all_zones' => ShippingZone::all(),
                'form_method' => 'GET',
            ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $model = new Model;
        $resource_name = self::$route_name;
        $action = route(
            "{$resource_name}.store",
            [$model->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'shop::admin.shipping.flat_rate.form',
                'name' => $resource_name,
                'item' => $model,
                'node' => null,
                'shipping_zones' => ShippingZone::all(),
                'form_action' => $action,
                'form_method' => 'POST',
                'route_params' => null,
            ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request current request
     * @param Model $flat_rate Rate to save
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, Model $model)
    {
        $this->validateRequest($request);

        $model->fill($request->all());
        $model->value *= 100;
        $model->min_basket_value *= 100;
        $model->max_basket_value *= 100;
        $model->save();

        $request->session()->flash('success', 'Flat rate updated for ' . $model->title);
        return redirect(route(...[
            self::$route_name . ".index"
        ]));

    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Model $flat_rate Rate to edit
     * @return \Illuminate\View\View
     */
    public function edit( Model $model)
    {
        $resource_name = self::$route_name;
        $action = route(
            "{$resource_name}.update",
            [$model->id]
        );
        return view('core::admin.builder.item')
            ->with([
                'view' => 'shop::admin.shipping.flat_rate.form',
                'name' => $resource_name,
                'item' => $model,
                'shipping_zones' => ShippingZone::all(),
                'node' => null,
                'form_action' => $action,
                'form_method' => 'PUT',
                'route_params' => null,
            ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request current request
     * @param Model $flat_rate Rate to update
     * @return \Illuminate\Http\Response
     */
    public function update( Request $request, Model $flat_rate)
    {
        $this->validateRequest($request);

        $flat_rate->fill($request->all());
        $flat_rate->value *= 100;
        $flat_rate->min_basket_value *= 100;
        $flat_rate->max_basket_value *= 100;
        $flat_rate->save();

        $request->session()->flash('success', 'Flat rate updated for ' . $flat_rate->title);
        return redirect(route(
            self::$route_name . ".index"
        ));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Request $request current request
     * @param Model $flat_rate Rate to remove
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Model $flat_rate)
    {
        $flat_rate->delete();

        if ($request->ajax()) {
            return response('success', 200);
        }

        $request->session()->flash('success', "'{$flat_rate->title}' delivery method has been deleted.");
        return redirect()->back();
    }

    /**
     * Trigger validation for request
     *
     * @param Request $request submitted request
     */
    private function validateRequest(Request $request)
    {
        $this->validate($request, [
            'title' => 'required',
            'shipping_zone_id' => 'required|numeric|exists:shipping_zones,id',
            'value' => 'required|numeric|min:0',
            'min_basket_value' => 'required|numeric|min:0',
            'max_basket_value' => 'required|numeric|min:' . $request->input('min_basket_value')
        ]);

    }
}
