<?php
/**
 * Price Class
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\PriceMethods;

use Config;
use Mtc\Shop\Contracts\BasketContract;

/**
 * Price Class
 *
 * @category Mtc\Shop\PriceMethods
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Price
{
    /**
     * The default price to show
     *
     * @var integer
     */
    public $price = 0;

    /**
     * The price without any discounts
     *
     * @var integer
     */
    public $price_original = 0;

    /**
     * The default price, without any tax if applicable
     *
     * @var integer
     */
    public $price_excluding_tax = 0;

    /**
     * The default price, including tax if applicable
     *
     * @var integer
     */
    public $price_including_tax = 0;
    /**
     * The default price, including tax if applicable
     *
     * @var integer
     */
    public $tax_on_price = 0;

    /**
     * The config array set within config/tax
     *
     * @var array
     */
    protected $config = [];

    /**
     * The currently set tax rate, and it's applicable rates.
     *
     * @var array
     */
    protected $tax_rates = [];

    /**
     * Build a new Price instance, setting the defined attributes.
     *
     * @param int    $price    Price
     * @param string $tax_rate Key of rate from tax.rates
     */
    public function __construct(int $price, $tax_rate = 'Standard')
    {
        // Get config for tax
        $this->config = Config::get('tax', []);

        // Get the rates for the specified key.
        if (array_key_exists($tax_rate, $this->config['rates'])) {
            $this->tax_rates = $this->config['rates'][$tax_rate];
        }

        // Remove tax from prices entered with tax.
        $this->price_excluding_tax = $price = $this->deductTax($price);

        // Add tax to prices
        $this->price_including_tax = $this->addTax($price);
        $this->price_original = $this->price_including_tax;
        if ($this->config['display_product_tax'] === false) {
            $this->price_original = $this->price_excluding_tax;
        }

        $this->price = $this->priceModifiers();
    }

    /**
     * Remove tax from the provided price if required.
     *
     * @param  int $price Price Value
     * @return int
     */
    protected function deductTax(int $price) : int
    {
        $this->tax_on_price = 0;
        $tax_rate = (float)$this->getTaxRate(Config::get('shop.country', 'GB'));

        if (true === $this->config['enabled']
            && true === $this->config['price_entered_with_tax']
            && $tax_rate > 0
        ) {
            $this->tax_on_price = $price - ( $price / ( $tax_rate + 1 ) );
        }

        return $price - $this->tax_on_price;
    }

    /**
     * Add tax to the provided price if required.
     *
     * @param  int $price Price value
     * @return int
     */
    protected function addTax(int $price) : int
    {
        $tax_amount = 0;
        if (true === $this->config['enabled']) {
            $tax_amount = $price * $this->getTaxRate();
        }
        return $price + $tax_amount;
    }

    /**
     * Deduct any discounts and the like from the orice.
     *
     * @return int
     */
    protected function priceModifiers() : int
    {
        // @todo make the event and the magic.
        return $this->price_original;
    }

    /**
     * Get the tax rate for the specific product.
     *
     * @param $country The key to use to termine tax rate
     *
     * @return float
     */
    protected function getTaxRate($country = null) : float
    {

        // Determine the user's country.
        if ($country === null) {
            $basket = \App::make(BasketContract::class);
            $address = $basket->addresses->sortBy('type')->first();
            $country = $address ? $address->country: Config::get('shop.country', 'GB');
        }

        // return (reset($this->tax_rates))['rate'];

        // Check if tax rate applies against this country.
        foreach ($this->tax_rates as $rate) {
            if ($rate['country'] == $country) {
                return $rate['rate'];
            }
        }

        return 0;
    }
}
