<?php

use Faker\Factory;
use Illuminate\Database\Seeder;
use Mtc\Shop\Product;
use Mtc\Shop\Order\Order;
use Mtc\Shop\Order\OrderStatus;

/**
 * Seed orders table with dummy data.
 *
 *
 * PHP Version 7
 *
 * @author   Haroldas Latonas <haroldas.latonas@mtcmedia.co.uk>
 */
class OrderSeeder extends Seeder
{

    protected $factory  = null;
    protected $order    = null;
    protected $total    = 0;

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $this->factory = Factory::create();

        $amount = false;
        while ($amount === false) {
            $amount = $this->command->ask("How many orders?", 500);
            $amount = filter_var($amount, FILTER_VALIDATE_INT);
        }

        $bar = $this->command->getOutput()->createProgressBar($amount);

        for ($i = 0; $i < $amount; $i++) {
            $this->addOrder();
            $bar->advance();
        }

        $bar->finish();
    }

    /**
     * Create order object and populate with dummy data
     *
     * @return void
     */
    protected function addOrder()
    {

        $payment_response = [
            "VendorTxCode" => "protxross-20170127132930-25",
            "VPSTxId" => "{0CCE87F1-6CB7-29F9-32A2-66E0BF8E84B5}",
            "Status" => "OK",
            "StatusDetail" => "0000 : The Authorisation was Successful.",
            "TxAuthNo" => "13818122",
            "AVSCV2" => "SECURITY CODE MATCH ONLY",
            "AddressResult" => "NOTMATCHED",
            "PostCodeResult" => "NOTMATCHED",
            "CV2Result" => "MATCHED",
            "GiftAid" => "0",
            "3DSecureStatus" => "OK",
            "CAVV" => "AAABARR5kwAAAAAAAAAAAAAAAAA=",
            "CardType" => "VISA",
            "Last4Digits" => "0006",
            "DeclineCode" => "00",
            "ExpiryDate" => "0120",
            "Amount" => "70.20",
            "BankAuthCode" => "999777\u0005\u0005\u0005\u0005\u0005"
        ];

        $orderStatus = collect(OrderStatus::all())->random();

        $this->order = new Order([
            'basket_id'         => null,
            'order_status_id'   => $orderStatus,
            'created_at'        => $this->factory->dateTimeBetween('-3 years', 'now'),
            'is_paid'           => rand(0,1),
            'email'             => $this->factory->safeEmail(),
            'phone'             => $this->factory->e164PhoneNumber(),
            'payment'           => $payment_response,
            'meta'              => ['shipping' => true],
            'payment_type'      => 'Sagepay Form'
        ]);

        $this->order->save();

        $this->order->notes()->create([
            'message'   => "Sagepay Form payment completed. The reference number is " . $payment_response['VPSTxId']
        ]);

        $this->addAddress();
        $this->orderItems();
    }
    /**
     * Creates address objects and assigns to order object
     *
     * @return void
     */
    protected  function addAddress() {
        $full_name = $this->factory->firstName . " " . $this->factory->lastName;
        $address = [
            'full_name'     => $full_name,
            'first_name'    => $this->factory->firstName,
            'last_name'     => $this->factory->lastName,
            'address1'      => $this->factory->streetAddress,
            'address2'      => $this->factory->secondaryAddress,
            'city'          => $this->factory->city,
            'state'         => $this->factory->state,
            'postcode'      => $this->factory->postcode,
            'country'       => $this->factory->country
        ];

        $address['type'] = "billing";
        $this->order->addresses()->create($address);
        $address['type'] = 'shipping';
        $this->order->addresses()->create($address);
    }

    /**
     * Creates and populates OrderItems object with random products
     * Calculates total value of Order
     *
     * @return void
     */
    protected function orderItems() {

        Product::all()->take($this->factory->numberBetween(1,4))->each(function($item){
            $variant = collect($item->variants->all())
                ->random();
            $quantity = rand(1,7);

            $this->order->items()->create([
                'code'          => $variant->code,
                'product_id'    => $variant->product_id,
                'product_title' => $variant->product->node->title,
                'variant_id'    => $variant->id,
                'variant_title' => $variant->node->title,
                'quantity'      => $quantity,
                'price'         => $variant->prices->multiple($quantity)->price,
            ]);
            $this->total += $variant->prices->multiple($quantity)->price * $quantity;
        });
        $this->order->total = $this->total;
        $this->order->save();
        $this->total = 0;
    }

}
