<?php

namespace Mtc\Shop\Http\Controllers\Admin;

use Mtc\Core\Http\Controllers\Controller;
use Event;
use Illuminate\Http\Request;
use League\Csv\Writer;
use Mtc\Core\Admin\Builder;
use Mtc\Core\Admin\ItemBuilder;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Order\OrderNote;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, OrderContract $order)
    {
        $query = $order->query()
            ->whereIsPaid(1)
            ->buildSearch($request)
            ->orderBy('id', 'desc');

        // Bring in search
        Event::listen(
            'core.builder.shop.admin.orders.head', function () {
                return 'shop::admin.order.index_head';
            }
        );

        // Bring in export
        Event::listen(
            'core.builder.shop.admin.orders.buttons', function () {
                return 'shop::admin.order.index_export';
            }
        );

        return (
            new Builder('shop.admin.orders', $query))
            ->columns(
                [
                    'id' => trans('fields.id'),
                    'name' => trans('shop::basket.personal.name'),
                    'email' => trans('shop::basket.personal.email'),
                    'status.title' => 'Status',
                    'date' => 'Date',
                ]
            )
            ->data(
                [
                    'name' => function ($order) {
                        $first = $order->addresses->first()->first_name;
                        $last = $order->addresses->first()->last_name;
                        return "{$first} {$last}";
                    },
                    'date' => function ($order) {
                        return $order->created_at;
                    },
                    'status.title' => function ($order) {
                        return $order->status->title;
                    }
                ]
            )
            ->view();
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param OrderContract $order Order Model
     *
     * @return \Illuminate\View\View
     */
    public function edit(OrderContract $order)
    {
        $title = "Order {$order->id}";
        $subtitle = $order->created_at->toDayDateTimeString();

        // Shipping method is stored as json, lets decode it
        $order->shipping_method = json_decode($order->shipping_method);

        // Only retrieve information for addresses we need and convert to a string
        $addresses = $order->addresses->keyBy('type')
            ->map(function ($address) {
                // Generate an address with each filled in var as new line
                $address['full'] = collect($address)
                    ->only([
                        'address1',
                        'address2',
                        'city',
                        'state',
                        'postcode',
                        'country'
                    ])
                    ->reject(function ($item) {
                        return empty(trim($item));
                    })
                    ->implode(PHP_EOL);

                return $address;
            });

        return view('shop::admin.order.edit')
            ->with(compact('order', 'title', 'subtitle', 'addresses'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param int                      $id      Order ID
     *
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \Illuminate\Http\Request $request Incoming Request
     * @param int                      $id      Order ID
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Add a note via the admin area for this order.
     *
     * @param OrderContract $order   Order
     * @param Request       $request Request with 'note' param.
     * @param OrderNote     $note    Blank OrderNote to populate
     *
     * @return \Illuminate\Http\Response Redirect to order page
     */
    public function addNote(OrderContract $order, Request $request, OrderNote $note)
    {
        $this->validate(
            $request, [
            'note' => 'required'
            ]
        );

        $note->user_id = $request->user()->id;
        $note->message = $request->input('note');

        $order->notes()->save($note);

        return redirect(route('shop.admin.orders.edit', $order) . '#orderNotes');
    }

    /**
     * Export of Orders to CSV
     *
     * @param  Request       $request
     * @param  OrderContract $order
     * @return void          Data sent to browser as a CSV
     */
    public function export(Request $request, OrderContract $order)
    {
        $query = $order->query()
            ->whereIsPaid(1)
            ->buildSearch($request)
            ->orderBy('id', 'desc');

        $csv = Writer::createFromFileObject(new \SplTempFileObject());

        // Add Titles
        $csv->insertOne([
            'Order ID',
            'Payment Gateway',
            'Date',
            'Customer Name',
            'Email',
            'Total Order Value',
            'Billing Country',
        ]);

        // Add Data
        $csv->insertAll(
            $query->get()
                ->map(function($order) {
                    return [
                        'id' => $order->id,
                        'gateway' => $order->payment_type,
                        'date' => $order->created_at->toDateTimeString(),
                        'name' => $order->addresses->first()->full_name,
                        'email' => $order->email,
                        'total' => $order->total / 100,
                        'billing country' => $order->addresses->first()->country,
                    ];
                })
                ->toArray()
        );

        $csv->output('orders_' . date('Y-m-d_H-i-s') . '.csv');
        exit;
    }
}
