<?php
/**
 * Product Controller (Public)
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Shop\Product;

/**
 * Control this display of products on the public site.
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ProductController extends Controller
{
    /**
     * Show the product.
     *
     * @param Request $request incoming request
     * @param int $id Product ID
     *
     * @return \Illuminate\View\View
     */
    public function show(Request $request, $id)
    {
        $product = Product::with([
            'media',
            'variants',
            'variants.node'
        ])->findOrFail($id);

        $this->setProductBreadCrumbs($product);

        foreach ($product->variants as $key => $variant) {
            if (!$variant->isInStock()) {
                // don't allow out of stock variants to be included
                unset($product->variants[$key]);
            }
        }

        $image = 'http://placehold.it/500x500';
        if ($product->media->isEmpty() === false) {
            $image = asset('storage/' . $product->media->first()->getSize('large'));
        }

        return view('shop::public.product')->with(compact('product', 'image'));
    }

    /**
     * Set up the array for breadcrumbs of this product.
     * Extended to its own method due to reduce complexity of the index method
     *
     * @param Product $product Loaded product
     */
    private function setProductBreadCrumbs(Product $product)
    {
        // Add breadcrumb for browse page
        $this->breadcrumbs->addCrumb(trans('core::text.shop'), action('\\' . BrowseController::class . '@index'));

        // Add browse page breadcrumbs
        if (!empty(session('user_journey.browse.taxonomies'))) {
            // check if user has come to this item from browse page
            // If so we need to make sure the correct taxonomy is selected to reflect the user journey
            $taxonomy = $product->node
                ->taxonomies
                ->reject(function ($taxonomy) {
                    return session('user_journey.browse.taxonomies')
                            ->whereIn('id', $taxonomy->ancestorsAndSelf()->pluck('id'))
                            ->count() == 0;
                })
                ->first();
        }

        // If no session taxonomy or taxonomies have changed due to multiple tabs
        if (empty($taxonomy)) {
            // If user has come to item directly we need to use the first taxonomy that exists for item
            $taxonomy = $product->node
                ->taxonomies
                ->first();
        }

        // Build breadcrumbs based on product taxonomies if they exist
        if ($taxonomy) {
            $crumb_trail = collect([
                $taxonomy
            ]);

            while ($taxonomy->depth > 1) {
                $taxonomy = $taxonomy->parent;
                $crumb_trail->prepend($taxonomy);
            }

            $crumb_trail->each(function ($taxonomy) {
                // Add a breadcrumb with the browse url
                $this->breadcrumbs->addCrumb($taxonomy->title,BrowseController::generateUrl([$taxonomy], 'absolute'));
            });
        }

        // Add breadcrumb for this page
        $this->breadcrumbs->addCrumb($product->node->title, '');

    }
}
