<?php
/**
 *
 * AddReservedItem
 *
 * PHP Version <7>
 *
 * @category Mtc\Shop\Listeners
 * @author   Joe Pritchard <joe.pritchard@mtcmedia.co.uk>
 *
 * Created: 11/04/2017 15:05
 *
 */

namespace Mtc\Shop\Listeners;

use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Events\ItemAddedToBasket;
use Mtc\Shop\Stock\Reserved as StockReserved;
use Mtc\Shop\Stock\Settings as StockSettings;
use Mtc\Shop\Variant;


/**
 * Class AddReservedItem
 *
 * @package Mtc\Shop\Listeners
 */
class AddReservedItem
{
    /**
     * Handle the ItemAddedToBasket event
     *
     * @param BasketContract $basket
     * @param Variant        $variant
     * @param int            $quantity
     *
     * @return void
     */
    public function handle(ItemAddedToBasket $event)
    {
        $basket   = $event->basket;
        $variant  = $event->variant;
        $quantity = $event->quantity;

        if ($variant->product->stock_enabled || $variant->stock_enabled) {
            $stocksettings = StockSettings::first();

            // we exclude the current variant because its quantity is not relevant to us,
            // since we're updating it in the next step
            $total_basket_reserved_items = StockReserved::current()
                ->where('basket_id', $basket->id)
                ->where('variant_id', '!=', $variant->id)
                ->selectRaw('SUM(quantity) as sumtotal')
                ->pluck('sumtotal')[0] ?? 0;

            if ($total_basket_reserved_items + $quantity <= config('stock.stock_max_reserved')) {
                // reserving this many units will not push us over our max, so:
                $variant_reserved = $variant->stockReserved()
                    ->current()
                    ->firstOrNew(['basket_id' => $basket->id]);

                $variant_reserved->quantity = $quantity;
                $variant_reserved->save();
            }

            // and while we're here, let's get rid of old reservations
            StockReserved::outOfDate()->delete();

        }
    }
}