<?php
/**
 * Product Model
 *
 * PHP Version 7
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop;

use Cache;
use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Media;
use Mtc\Core\Node;
use Mtc\Core\Nodeable;
use Mtc\Shop\Http\Controllers\ProductController;
use Mtc\Shop\PriceMethods\PerProduct;

/**
 * Base Product Model
 *
 * @category Mtc\Shop
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class Product extends Model
{
    use Nodeable;
    use Pricing;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'price_method',
        'stock_enabled',
        'stock_status',
        'tax',
    ];

    /**
     * The attributes to load with this object when instatiating
     *
     * @var array
     */
    protected $with = ['node'];

    /**
     * Default attributes for a new Product.
     *
     * @var array
     */
    protected $attributes = [
        'price_method' => PerProduct::class,
        'stock_enabled' => false,
        'tax' => 'Standard',
    ];

    protected $touches = [
        'node',
    ];

    /**
     * Name of the controller that handles public requests for this object
     * @var string $public_controller
     */
    public static $public_controller = ProductController::class . "@show";

    /**
     * The attributes that should be loaded when the model is loaded.
     *
     * @var array
     */
    protected $appends = ['price'];

    /**
     * Retrieve the related variants (sizes)
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function variants()
    {
        return $this->hasMany(Variant::class);
    }

    /**
     * Retrieve any related media.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function media()
    {
        return $this->morphMany(Media::class, 'parent')
            ->orderBy('order', 'asc');
    }

    /**
     * function isInStock
     *
     * A product is considered in stock if
     *  - Stock management is disabled globally
     *  - Stock management is disabled for this product
     *  - Stock management is disabled for any of this product's variants
     *  - The sum of all variant's stock levels is greater than zero
     *
     * @return bool
     */
    public function isInStock() {

        $variants_stock_quantity = 0;
        foreach ($this->variants as $variant) {
            $variants_stock_quantity += $variant->stock_quantity;
        }

        return $variants_stock_quantity > 0;
    }

    /**
     * Scope for Eloquent queries to search parent node for a term.
     *
     * @param \Illuminate\Eloquent\Query\Builder $query Query Builder
     * @param string                             $term  Search Query
     *
     * @return \Illuminate\Eloquent\Query\Builder
     */
    public function scopeSearch($query, $term)
    {
        if (!empty($term)) {
            $query->whereHas(
                'node', function ($query) use ($term) {
                    $query->search($term);
                }
            );
        }
        return $query;
    }

    /**
     * Return a URL for the product.
     *
     * @param boolean                $absolute Include the Site's URL.
     * @param boolean|\Mtc\Core\Node $node     The associated node if available
     *
     * @return string
     */
    public function getUrl($absolute = true, $node = false)
    {
        // As this may be called ahead of the node being saved, we need to
        // pass through.
        if ($node instanceof Node === false) {
            $node = $this->node;
        }

        return route(
            'shop.product',
            [$this, $node->slug],
            $absolute
        );
    }

    /**
     * Build a search query for the admin panel.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query   Model Query
     * @param \Illuminate\Http\Request              $request Browser Request
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeBuildSearch(
        \Illuminate\Database\Eloquent\Builder $query,
        \Illuminate\Http\Request $request
    ) {
        if ($request->has('title')) {
            $query->whereHas(
                'node', function ($query) use ($request) {
                    $query->where('title', 'LIKE', '%' . $request->input('title') . '%');
                }
            );
        };

        return $query;
    }
}
