<?php
/**
 * Product Repository
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Repositories
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
namespace Mtc\Shop\Repositories;

use Cache;
use Illuminate\Http\Request;
use Mtc\Core\Node;
use Mtc\Core\Taxonomy;
use Mtc\Shop\Contracts\ProductRepositoryContract;
use Mtc\Shop\Product;

/**
 * Used to handle more complex queries on products.
 *
 * @category Mtc\Shop\Repositories
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class ProductRepository implements ProductRepositoryContract
{
    /**
     * The query builder instance
     *
     * @var null|\Illuminate\Database\Eloquent\Builder
     */
    protected $query = null;

    /**
     * Run a search on the provided request.
     *
     * @param Request $request Incoming filter request
     *
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function filter(Request $request)
    {
        if (empty($request->all())) {
            return Cache::remember(
                'product_repo_filter', 5, function () {
                    return $this->search(null)
                        ->baseQuery()
                        ->sortBy(null)
                        ->paginate();
                }
            );
        }

        return $this->search($request->input('query'))
            ->baseQuery()
            ->withTaxonomy($request->input('selected'))
            ->sortBy($request->input('sort_by'))
            ->paginate();
    }

    /**
     * Run a search on Product Nodes matching the provided query, storing the
     * update query within $this->query.
     *
     * @param string $query Search request
     *
     * @return self
     */
    protected function search($query)
    {
        if (empty($query)) {
            $this->query = Node::query();
        } else {
            $this->query = Node::whereIn(
                'nodes.id',
                Node::search($query)
                    ->where('nodeable_type', Product::class)
                    ->get()
                    ->pluck('id')
            );
        }

        return $this;
    }

    /**
     * Build a select query pulling out only the basic information required for
     * search, joining the Node and Product tables and only showing products
     * that are visible.
     *
     * @return self
     */
    protected function baseQuery()
    {
        $this->query->select(
            [
                    'nodes.id',
                    'nodes.title',
                    'nodes.description',
                    'nodes.slug',
                    'nodes.nodeable_id',
                    'nodes.nodeable_type',
                    'po.sort_price as price',
                    'nodes.url',
                ]
        )->join(
            'products as po',
            'nodes.nodeable_id',
            '=',
            'po.id'
        )->with(['nodeable', 'nodeable.media'])
            ->where('nodeable_type', Product::class)
            ->public();

        return $this;
    }

    /**
     * Continue building the query to determine if any of the provided
     * taxonomies have been attached to a Node.
     *
     * @param array $selected List of Taxonomy IDs
     *
     * @return self
     */
    protected function withTaxonomy($selected)
    {
        $selected = collect($selected)->pluck('id');
        $selected = Taxonomy::whereIn('id', $selected)->get();
        $selected = $selected->reject(
            function ($item) use ($selected) {
                foreach ($selected as $option) {
                    if ($item->isAncestorOf($option)) {
                        return true;
                    }
                }

                return false;
            }
        )->map(
            function ($item) {
                    // Return a list of these taxonomy ids and their children
                    return \DB::table('taxonomies')
                        ->where(function ($query) use ($item) {
                                $query->where('lft', '>=', $item->lft)
                                    ->where('rgt', '<=', $item->rgt);
                            }
                        )
                        ->pluck('id');
            }
        )->collapse()
        ->toArray();

        if (is_array($selected) && !empty($selected)) {
            $this->query->join(
                'node_taxonomy as nt',
                'nt.node_id',
                '=',
                'nodes.id'
            )->whereIn('nt.taxonomy_id', $selected);
        }

        return $this;
    }

    /**
     * Determine the sort method to put in place.
     *
     * @param string $method Sort method key
     *
     * @return self
     */
    protected function sortBy($method)
    {
        switch($method) {
        default:
        case 'new-additions':
            $this->query->orderBy('nodes.created_at', 'desc');
            break;
        case 'price-asc':
            $this->query->orderBy('po.sort_price', 'asc');
            break;
        case 'price-desc':
            $this->query->orderBy('po.sort_price', 'desc');
            break;
        }

        return $this;
    }

    /**
     * Return a paginated list of query results.
     *
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    protected function paginate()
    {
        return $this->query->paginate();
    }
}
