<?php
/**
 * Basket Controller
 *
 * PHP Version 7
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */

namespace Mtc\Shop\Http\Controllers;

use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Mtc\Core\Country;
use Mtc\Shop\Contracts\BasketContract;
use Mtc\Shop\Contracts\OrderContract;
use Mtc\Shop\Events\RetrievePaymentGateways;
use Mtc\Shop\Http\Requests\StoreBasketInfo;
use Mtc\Shop\Http\Requests\StoreBasketItems;
use Mtc\Shop\Variant;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Asctions revolving around Basket creation and update.
 *
 * @category Mtc\Shop\Http\Controllers
 * @package  Mtc\Shop
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class BasketController extends Controller
{
    /**
     * Display the basket.
     *
     * @param Request        $request Incoming request used to check if wanting JSON.
     * @param BasketContract $basket  Basket model
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise View.
     */
    public function index(Request $request, BasketContract $basket)
    {
        $basket->items = $this->getProcessedItems($basket);
        $basket->cost_subtotal = $basket->cost_total = $basket->items->sum('price_total');
        $basket->action = route('shop.basket.store');
        $basket->editable = true;

        if ($request->wantsJson()) {
            return $basket;
        }

        // Get a list of countries, with the default on top.
        $countries = Country::whereStatus(1)->get()
            ->sortBy(
                function($item, $index) {
                    if ($item->code == env('DEFAULT_COUNTRY', 'GB')) {
                        $index = -1;
                    }

                    return $index;
                }
            );

        return view('shop::public.basket')
            ->with(compact('basket', 'countries'));
    }

    /**
     * Save items to a basket.
     *
     * @param StoreBasketItems $request Validated Request
     * @param BasketContract   $basket  Basket Model
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested
     *                                                         or just a view.
     */
    public function store(StoreBasketItems $request, BasketContract $basket)
    {
        foreach ($request->input('items') as $item) {
            $variant = Variant::find($item['variant_id']);
            $basket->addItem($variant, $item['quantity']);
        }

        $basket->items()->where('quantity', '<', 1)->delete();


        $request->session()->reflash();
        $request->session()->flash('previous', $variant->product->getUrl());

        return $this->index($request, $basket);
    }

    /**
     * Update a basket with address + other information.
     *
     * @param StoreBasketInfo $request Validated Request
     * @param BasketContract  $basket  Basket Model
     *
     * @return \Illuminate\Http\Response
     */
    public function update(StoreBasketInfo $request, BasketContract $basket)
    {
        $basket->fill($request->only(['email', 'phone']));

        $billing = $basket->addresses()
            ->firstOrNew(['type' => 'billing'])
            ->fill($request->input('addresses.billing', []));
        $billing->first_name = $request->input('first_name');
        $billing->last_name = $request->input('last_name');
        $billing->save();

        if (true === filter_var($request->input('shipping'), FILTER_VALIDATE_BOOLEAN)) {
            $shipping = $basket->addresses()
                ->firstOrNew(['type' => 'shipping'])
                ->fill($request->input('addresses.shipping', []))
                ->save();
            $basket->meta = ['shipping' => true];
        } else {
            $basket->addresses()->whereType('shipping')->delete();
            $basket->meta = ['shipping' => false];
        }

        $basket->save();

        return redirect(route('shop.basket.overview'));
    }

    /**
     * Return a list of items associated with this basket with their URL,
     * price per unit and total price per line (excluding tax)
     *
     * @param BasketContract $basket Basket model
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    protected function getProcessedItems(BasketContract $basket)
    {
        return $basket->items()
            ->with(
                [
                'variant',
                'variant.node',
                'variant.product',
                'variant.product.node',
                ]
            )->get()->map(
                function($line) {
                        $line->url = $line->variant->product->getUrl();
                        $line->price_unit = $line->variant->price->perVariant($line->quantity);
                        $line->price_total = $line->price_unit * $line->quantity;
                        return $line;
                }
            );
    }

    /**
     * Display the basket overview page, where customers are able to choose
     * their selected payment gateway.
     *
     * @param Request        $request Default Request
     * @param BasketContract $basket  Basket model
     *
     * @return \Illuminate\Http\Response|\Illuminate\View\View JSON if requested, otherwise just a view. Redirect
     *                                                         will occur if basket is empty.
     */
    public function overview(Request $request, BasketContract $basket)
    {
        if ($basket->items->isEmpty()) {
            return redirect(route('shop.basket.index'));
        }

        // Generate a new order from the current basket.
        $order = resolve(OrderContract::class)->createFromBasket($basket);

        // Set default editable state for Vue.s
        $basket->editable = false;
        $basket->items = $this->getProcessedItems($basket);

        if ($request->wantsJson()) {
            return $basket;
        }

        $addresses = $basket->addresses
            ->keyBy('type');

        if (!$addresses->has('shipping')) {
            $addresses['shipping'] = $addresses['billing'];
        }

        // Get a list of registered payment gateways.
        $payment_gateways = event(new RetrievePaymentGateways($order));

        $request->session()->put('order_id', $order->id);

        return view('shop::public.overview')
            ->with(compact('basket', 'addresses', 'order', 'payment_gateways'));
    }

    /**
     * Send the user to their requested payment gateway to complete their order.
     *
     * @param Request       $request Default request.
     * @param OrderContract $order   Order model
     *
     * @return Illumnate\Http\Response Redirect to payment route or external URL
     */
    public function confirm(Request $request, OrderContract $order)
    {
        $order_id = $request->session()->get('order_id');
        $order = $order->find($order_id);

        if (!$order) {
            return redirect(route('shop.basket.overview'));
        }

        $class = $request->input('payment_method');
        $gateway = new $class;
        $url = $gateway->getTransactionUrl($order);
        return redirect($url);
    }
}
